
///////////////////////////////////////////////////////////
//                                                       //
//                         SAGA                          //
//                                                       //
//      System for Automated Geoscientific Analyses      //
//                                                       //
//                     Tool Library                      //
//                       Tutorial                        //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//                 Grid_Value_Request.cpp                //
//                                                       //
//                 Copyright (C) 2003 by                 //
//                      Olaf Conrad                      //
//                                                       //
//-------------------------------------------------------//
//                                                       //
// This file is part of 'SAGA - System for Automated     //
// Geoscientific Analyses'. SAGA is free software; you   //
// can redistribute it and/or modify it under the terms  //
// of the GNU General Public License as published by the //
// Free Software Foundation, either version 2 of the     //
// License, or (at your option) any later version.       //
//                                                       //
// SAGA is distributed in the hope that it will be       //
// useful, but WITHOUT ANY WARRANTY; without even the    //
// implied warranty of MERCHANTABILITY or FITNESS FOR A  //
// PARTICULAR PURPOSE. See the GNU General Public        //
// License for more details.                             //
//                                                       //
// You should have received a copy of the GNU General    //
// Public License along with this program; if not, see   //
// <http://www.gnu.org/licenses/>.                       //
//                                                       //
//-------------------------------------------------------//
//                                                       //
//    e-mail:     oconrad@saga-gis.org                   //
//                                                       //
//    contact:    Olaf Conrad                            //
//                Institute of Geography                 //
//                University of Goettingen               //
//                Goldschmidtstr. 5                      //
//                37077 Goettingen                       //
//                Germany                                //
//                                                       //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
#include "Grid_Value_Request.h"

//---------------------------------------------------------
#define	FIELD_NAME		0
#define	FIELD_VALUE		1

#define	FIELD_X_WORLD	0
#define	FIELD_Y_WORLD	1
#define FIELD_X_GRID	2
#define FIELD_Y_GRID	3
#define	FIELD_GRIDS		4


///////////////////////////////////////////////////////////
//														 //
//														 //
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
CGrid_Value_Request::CGrid_Value_Request(void)
{
	Set_Name		(_TL("Grid Value Request"));

	Set_Author		("O.Conrad (c) 2003");

	Set_Description	(_TW(
		"The tool reads out grids values from one or more input grids and reports these in a table. "
		"Besides the grid value(s), the x- and y-coordinates of the queried cell(s) are reported. The "
		"reported grid coordinates (column, row) are those of the first input grid.\n"
	));

	//-----------------------------------------------------
	m_pGrids = Parameters.Add_Grid_List("",
		"GRID_LIST"   , _TL("Grids to request"),
		_TL(""),
		PARAMETER_INPUT
	)->asGridList();

	Parameters.Add_Table("",
		"VALUES"    , _TL("Values"),
		_TL(""),
		PARAMETER_OUTPUT
	);

	Parameters.Add_Choice("",
		"METHOD"    , _TL("Method"),
		_TL(""),
		CSG_String::Format(SG_T("%s|%s|"),
			_TL("Single value"),
			_TL("Collect values")
		), 0
	);

	Parameters.Add_Grid_Resampling("",
		"RESAMPLING", _TL("Resampling"),
		_TL("")
	);

	Set_Drag_Mode(TOOL_INTERACTIVE_DRAG_NONE);
}


///////////////////////////////////////////////////////////
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
bool CGrid_Value_Request::On_Execute(void)
{
	if( m_pGrids->Get_Grid_Count() > 0 )
	{
		m_Method = Parameters("METHOD")->asInt();

		m_pTable = Parameters("VALUES")->asTable();
		m_pTable->Destroy();
		m_pTable->Set_Name(_TL("Grid Values"));

		m_Resampling = Parameters("RESAMPLING")->asGrid_Resampling();

		switch( m_Method )
		{
		default:
			m_pTable->Add_Field(_TL("NAME" ), SG_DATATYPE_String);
			m_pTable->Add_Field(_TL("VALUE"), SG_DATATYPE_Double);

			m_pTable->Add_Record()->Set_Value(FIELD_NAME, _TL("X World"));
			m_pTable->Add_Record()->Set_Value(FIELD_NAME, _TL("Y World"));
			m_pTable->Add_Record()->Set_Value(FIELD_NAME, _TL("X Grid 1"));
			m_pTable->Add_Record()->Set_Value(FIELD_NAME, _TL("Y Grid 1"));

			for(int iGrid=0; iGrid<m_pGrids->Get_Grid_Count(); iGrid++)
			{
				m_pTable->Add_Record()->Set_Value(FIELD_NAME, m_pGrids->Get_Grid(iGrid)->Get_Name());
			}
			break;

		case  1:
			m_pTable->Add_Field(_TL("X World" ), SG_DATATYPE_Double);
			m_pTable->Add_Field(_TL("Y World" ), SG_DATATYPE_Double);
			m_pTable->Add_Field(_TL("X Grid 1"), SG_DATATYPE_Int);
			m_pTable->Add_Field(_TL("Y Grid 1"), SG_DATATYPE_Int);

			for(int iGrid=0; iGrid<m_pGrids->Get_Grid_Count(); iGrid++)
			{
				m_pTable->Add_Field(m_pGrids->Get_Grid(iGrid)->Get_Name(), SG_DATATYPE_Double);
			}
			break;
		}

		DataObject_Update(m_pTable, true);

		return( true );
	}

	Message_Dlg(_TL("There is no grid to request."), Get_Name());

	return( false );
}

//---------------------------------------------------------
bool CGrid_Value_Request::On_Execute_Position(CSG_Point ptWorld, TSG_Tool_Interactive_Mode Mode)
{
	if( m_pGrids->Get_Grid_Count() > 0 )
	{
		switch( m_Method )
		{
		//-------------------------------------------------
		default:
			if( Mode == TOOL_INTERACTIVE_LDOWN || Mode == TOOL_INTERACTIVE_MOVE_LDOWN )
			{
				m_pTable->Get_Record(FIELD_X_WORLD)->Set_Value(FIELD_VALUE, ptWorld.x);
				m_pTable->Get_Record(FIELD_Y_WORLD)->Set_Value(FIELD_VALUE, ptWorld.y);

				m_pTable->Get_Record(FIELD_X_GRID)->Set_Value(FIELD_VALUE, m_pGrids->Get_Grid(0)->Get_System().Get_xWorld_to_Grid(ptWorld.x));
				m_pTable->Get_Record(FIELD_Y_GRID)->Set_Value(FIELD_VALUE, m_pGrids->Get_Grid(0)->Get_System().Get_yWorld_to_Grid(ptWorld.y));

				for(int iGrid=0; iGrid<m_pGrids->Get_Grid_Count(); iGrid++)
				{
					double Value;

					if( m_pGrids->Get_Grid(iGrid)->Get_Value(ptWorld, Value, m_Resampling) )
					{
						m_pTable->Get_Record(iGrid + FIELD_GRIDS)->Set_Value(FIELD_VALUE, Value);
					}
					else
					{
						m_pTable->Get_Record(iGrid + FIELD_GRIDS)->Set_Value(FIELD_VALUE, 0.);
					}
				}

				DataObject_Update(m_pTable);

				return( true );
			}
			break;

		//-------------------------------------------------
		case  1:
			if( Mode == TOOL_INTERACTIVE_LDOWN )
			{
				CSG_Table_Record *pRecord = m_pTable->Add_Record();

				pRecord->Set_Value(FIELD_X_WORLD, ptWorld.x);
				pRecord->Set_Value(FIELD_Y_WORLD, ptWorld.y);

				pRecord->Set_Value(FIELD_X_GRID, m_pGrids->Get_Grid(0)->Get_System().Get_xWorld_to_Grid(ptWorld.x));
				pRecord->Set_Value(FIELD_Y_GRID, m_pGrids->Get_Grid(0)->Get_System().Get_yWorld_to_Grid(ptWorld.y));

				for(int iGrid=0; iGrid<m_pGrids->Get_Grid_Count(); iGrid++)
				{
					double Value;

					if( m_pGrids->Get_Grid(iGrid)->Get_Value(ptWorld, Value, m_Resampling) )
					{
						pRecord->Set_Value(FIELD_GRIDS + iGrid, Value);
					}
					else
					{
						pRecord->Set_Value(FIELD_GRIDS + iGrid, 0.);
					}
				}

				DataObject_Update(m_pTable);

				return( true );
			}
			break;
		}
	}

	return( false );
}


///////////////////////////////////////////////////////////
//														 //
//														 //
//														 //
///////////////////////////////////////////////////////////

//---------------------------------------------------------
