#!/bin/bash

set -e
. "$(dirname "$0")/../util/lib.sh"

init
check_hostaliases

rm -rf .data-primary .data-secondary .data-external .mail
rm -f {primary,secondary,external}/domains/*/dkim:*.pem

# Build with the DNS override, so we can fake DNS records.
export GOTAGS="dnsoverride"

# Two servers for the same domain "dodo":
# primary - listens on 127.0.0.10:1025
# secondary - listens on 127.0.0.11:1025
#
# One server for domain "kiwi":
# external - listens on 127.0.0.20:1025

CONFDIR=primary generate_certs_for primary
CONFDIR=primary add_user user111@dodo user111
CONFDIR=primary chasquid-util dkim-keygen --algo=ed25519 \
	dodo sel-primary-1 > /dev/null

CONFDIR=secondary generate_certs_for secondary
CONFDIR=secondary add_user user222@dodo user222
CONFDIR=secondary chasquid-util dkim-keygen --algo=ed25519 \
	dodo sel-secondary-1 > /dev/null

CONFDIR=external generate_certs_for external
CONFDIR=external add_user user333@kiwi user333
CONFDIR=external chasquid-util dkim-keygen --algo=ed25519 \
	kiwi sel-external-1 > /dev/null


# Launch minidns in the background using our configuration.
# Augment the zones with the dkim ones.
cp zones .zones
CONFDIR=primary   chasquid-util dkim-dns dodo | sed 's/"//g' >> .zones
CONFDIR=secondary chasquid-util dkim-dns dodo | sed 's/"//g' >> .zones
CONFDIR=external  chasquid-util dkim-dns kiwi | sed 's/"//g' >> .zones
minidns_bg --addr=":9053" -zones=.zones >> .minidns.log 2>&1


mkdir -p .logs

chasquid -v=2 --logfile=.logs/primary-chasquid.log --config_dir=primary \
	--testing__dns_addr=127.0.0.1:9053 \
	--testing__outgoing_smtp_port=1025 &
chasquid -v=2 --logfile=.logs/secondary-chasquid.log --config_dir=secondary \
	--testing__dns_addr=127.0.0.1:9053 \
	--testing__outgoing_smtp_port=1025 &
chasquid -v=2 --logfile=.logs/external-chasquid.log --config_dir=external \
	--testing__dns_addr=127.0.0.1:9053 \
	--testing__outgoing_smtp_port=1025 &

wait_until "true < /dev/tcp/127.0.0.10/1025" 2>/dev/null
wait_until "true < /dev/tcp/127.0.0.11/1025" 2>/dev/null
wait_until "true < /dev/tcp/127.0.0.20/1025" 2>/dev/null
wait_until_ready 9053

# Connect to secondary, send an email to user111@dodo (which exists only in
# the primary).  It should be forwarded to the primary.
# Note this also verifies SRS is correct (by comparing the Received headers),
# and that DKIM signatures are generated by secondary, and successfully
# validated by primary.
smtpc -c=smtpc-secondary.conf user111@dodo < content
wait_for_file .mail/primary:user111@dodo
mail_diff expected-primary-user111@dodo .mail/primary:user111@dodo

# Connect to the secondary, send an email to user333@kiwi (which exists only
# in external). It should be DKIM signed and delivered to the external server.
# This is a normal delivery.
smtpc -c=smtpc-secondary.conf user333@kiwi < content
wait_for_file .mail/external:user333@kiwi
mail_diff expected-external-user333@kiwi .mail/external:user333@kiwi

# Connect to the secondary, send an email to chain-1@dodo, which has a long
# alias chain:
#   secondary: chain-1-1@dodo  ->  chain-1-2@dodo via primary
#   primary:   chain-1-2@dodo  ->  chain-1-3@kiwi
#   external:  chain-1-3@kiwi  ->  chain-1-4@dodo via secondary
#   secondary: chain-1-4@dodo  ->  chain-1-5@dodo via primary
#   primary:   chain-1-5@dodo  ->  user111@dodo
rm .mail/primary:user111@dodo
smtpc -c=smtpc-secondary.conf chain-1-1@dodo < content
wait_for_file .mail/primary:user111@dodo
mail_diff expected-chain-1 .mail/primary:user111@dodo

success
