/*
 * Copyright (C) 2002-2008 Stefan Holst
 * Copyright (C) 2005-2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * oxine main program
 */

#define LOG

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xui_getopt.h"

#include <stdio.h>
#include <ctype.h>
#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <pthread.h>
#include <unistd.h>

#include <xine.h>
#include <xine/xmlparser.h>

#include "common.h"

#include "utils.h"
#include "oxine_event.h"
#include "oxine.h"
#include "odk.h"
#include "otk.h"
#include "mediamarks.h"
#include "playlist.h"
#include "../kbindings.h"
#include "../actions.h"
#include "../event.h"
#include "../videowin.h"
#include "../playlist.h"
#include "../mediamark.h"
#include "../xine-toolkit/cfg_parse.h"

#undef OXINE_TRANS_DUMMIES
/** in theory, we need to scan xml file at build time... */
#ifdef OXINE_TRANS_DUMMIES
static const char * const trans_dummies[] = {
  N_("Mediamarks"), N_("Playlist"), N_("Shutdown")
};
#endif

#ifdef OXINE_LOCK
static oxine_t *oxine_instance_get (oxine_t *oxine) {
  if (!oxine)
    return NULL;
  pthread_mutex_lock (&oxine->instance.mutex);
  oxine->instance.locks++;
  pthread_mutex_unlock (&oxine->instance.mutex);
  return oxine;
}

static void oxine_instance_unget (oxine_t *oxine) {
  if (!oxine)
    return;
  pthread_mutex_lock (&oxine->instance.mutex);
  if (0 == --oxine->instance.locks)
    pthread_cond_broadcast (&oxine->instance.unlocked);
  pthread_mutex_unlock (&oxine->instance.mutex);
}
#endif

typedef struct menuitem_s menuitem_t;
typedef enum {
  _MIT_NONE = 0,
  _MIT_autoplay,
  _MIT_mrl,
  _MIT_mediamarks,
  _MIT_playlist,
  _MIT_shutdown
} _menuitem_type_t;

struct menuitem_s {
  xitk_dnode_t node;
  oxine_t *oxine;
  char *title;
  void *data;
  int  x, y, w, h;
  _menuitem_type_t type;
};

typedef struct {
  oxine_session_t session;
  xitk_dlist_t menu_items;
} oxine_main_session_t;

static void oxine_main_close (oxine_t *oxine) {
  oxine->main_window = NULL;
  oxine->info_window = NULL;
  otk_clear (oxine->otk);
}

static int oxine_main_read (oxine_t *oxine, xitk_dlist_t *list, const char *mrl) {
  size_t size = 2 << 20;
  char *file = xitk_cfg_load (mrl, &size);
  xml_node_t *node;

  if (!file)
    return 0;

  xml_parser_init_R (xml_parser_t *xml, file, size, XML_PARSER_CASE_INSENSITIVE);

  if (xml_parser_build_tree_R (xml, &node) < 0) {
    lprintf ("xml parsing of %s failed\n", mrl);
    xml_parser_finalize_R (xml);
    return 0;
  }
  if (strcasecmp (node->name, "oxinemm")) {
    lprintf ("error, root node must be OXINEMM\n");
    xml_parser_finalize_R (xml);
    return 0;
  }

  node = node->child;
  if (!node || strcasecmp (node->name, "window")) {
    lprintf ("error, node WINDOW expected (%s found)\n", (!node) ? "(null)" : node->name);
    xml_parser_finalize_R (xml);
    return 0;
  }
  oxine->win_x = atoi (xml_parser_get_property (node, "x"));
  oxine->win_y = atoi (xml_parser_get_property (node, "y"));
  oxine->win_w = atoi (xml_parser_get_property (node, "width"));
  oxine->win_h = atoi (xml_parser_get_property (node, "height"));

  for (node = node->child; node; node = node->next) {
    if (!strcasecmp (node->name, "entry")) {
      xml_node_t *n2;
      menuitem_t *item = ho_new (menuitem_t);
      if (!item)
        break;
      xitk_dnode_init (&item->node);
      item->x = atoi (xml_parser_get_property (node, "x"));
      item->y = atoi (xml_parser_get_property (node, "y"));
      item->w = atoi (xml_parser_get_property (node, "width"));
      item->h = atoi (xml_parser_get_property (node, "height"));
      item->oxine = oxine;
      item->title = NULL;
      item->data = NULL;
      item->type = _MIT_NONE;
      for (n2 = node->child; n2; n2 = n2->next) {
        if (!strcasecmp (n2->name, "title")) {
          item->title = (char *)n2->data;
        } else if (!strcasecmp (n2->name, "action")) {
          const char *type = xml_parser_get_property (n2, "type");
          if (type) {
            if (!strcasecmp (type, "autoplay")) {
              item->data = (char *)xml_parser_get_property (n2, "parameter");
              item->type = _MIT_autoplay;
            } else if (!strcasecmp (type, "mrl")) {
              item->data = (char *)xml_parser_get_property (n2, "parameter");
              item->type = _MIT_mrl;
            } else if (!strcasecmp (type, "mediamarks")) {
              item->type = _MIT_mediamarks;
            } else if (!strcasecmp (type, "playlist")) {
              item->type = _MIT_playlist;
            } else if (!strcasecmp (type, "shutdown")) {
              item->type = _MIT_shutdown;
            } else if (!strcasecmp (type, "shell")) {
            }
          }
        }
      }
      if (item->title)
        item->title = ho_strdup (gettext (item->title));
      if (item->data)
        item->data = ho_strdup (item->data);
      xitk_dlist_add_tail (list, &item->node);
    }
  }

  xml_parser_free_tree (node);
  xml_parser_finalize_R (xml);
  xitk_cfg_unload (file);
  return 1;
}

static void oxine_main_menuitem_cb (void *data) {
  menuitem_t *item = (menuitem_t *)data;
  char *parameter = (char *)item->data;

  switch (item->type) {
    case _MIT_shutdown:
      item->oxine->shutdown = 1;
      break;
    case _MIT_mrl:
#ifdef OXINE_LOCK
      item->oxine = oxine_instance_get (item->oxine);
      if (!item->oxine)
        break;
#endif
      oxine_main_close (item->oxine);
      odk_open_and_play (item->oxine->odk, parameter);
#ifdef OXINE_LOCK
      oxine_instance_unget (item->oxine);
#endif
      break;
    case _MIT_autoplay:
      {
        int num_mrls, j;
#if XINE_MAJOR_VERSION < 1 || (XINE_MAJOR_VERSION == 1 && XINE_MINOR_VERSION < 2)
        char **autoplay_mrls;
#else
        const char * const *autoplay_mrls;
#endif
#ifdef OXINE_LOCK
        item->oxine = oxine_instance_get (item->oxine);
        if (!item->oxine)
          break;
#endif
        autoplay_mrls = xine_get_autoplay_mrls (item->oxine->gui->xine, parameter, &num_mrls);
        if (autoplay_mrls) {
          playlist_action (item->oxine->gui, PLAYLIST_CLEAR);
          for (j = 0; j < num_mrls; j++)
            gui_playlist_append (item->oxine->gui, (const char *)autoplay_mrls[j],
              (const char *)autoplay_mrls[j], NULL, 0, -1, 0, 0);
          oxine_main_close (item->oxine);
          gui_playlist_play (item->oxine->gui, 0);
        }
#ifdef OXINE_LOCK
        oxine_instance_unget (item->oxine);
#endif
      }
      break;
    case _MIT_playlist:
      oxine_win_main (item->oxine, oxine_win_id_playlist, oxine_win_action_show);
      break;
    case _MIT_mediamarks:
      oxine_win_main (item->oxine, oxine_win_id_mediamarks, oxine_win_action_show);
      break;
    default: ;
  }
}

static int oxine_main_main (oxine_t *oxine, oxine_win_id_t id, oxine_win_action_t action) {
  oxine_main_session_t *session;
  xitk_dnode_t *node;

  if (id != oxine_win_id_main)
    return 0;
  if (!oxine)
    return 0;
  xitk_container (session, oxine->sessions[oxine_win_id_main], session);
  if (!session)
    return 0;

  switch (action) {
    case oxine_win_action_show:
      if (!session->menu_items.head.next->next) {
        char mmpath[XITK_PATH_MAX + XITK_NAME_MAX];
        snprintf (mmpath, sizeof (mmpath),"%s/.xine/oxine/mainmenu", xine_get_homedir ());
        if (!oxine_main_read (oxine, &session->menu_items, mmpath)) {
          lprintf ("trying to load system wide mainmenu\n");
          snprintf (mmpath, sizeof (mmpath),"%s/mainmenu", XINE_OXINEDIR);
          if (!oxine_main_read (oxine, &session->menu_items, mmpath)) {
            xitk_dlist_init (&session->menu_items);
            return 0;
          }
        }
      }
      xitk_lock (oxine->xitk, 1);
      if (oxine->info_window) {
        otk_destroy_widget (oxine->info_window);
        oxine->info_window = NULL;
        otk_draw_all (oxine->otk);
        if (oxine->lines[0]) 
          ho_free (oxine->lines[0]);
        if (oxine->lines[1])
          ho_free (oxine->lines[1]);
        if (oxine->lines[2])
          ho_free (oxine->lines[2]);
      }
      xitk_lock (oxine->xitk, 0);
      oxine->pauseplay = NULL;
      oxine->main_window = otk_window_new (oxine->otk, NULL, oxine->win_x, oxine->win_y, oxine->win_w, oxine->win_h);
      for (node = session->menu_items.head.next; node->next; node = node->next) {
        menuitem_t *item;
        otk_widget_t *b;
        xitk_container (item, node, node);
        b = otk_button_new (oxine->main_window, item->x, item->y, item->w, item->h, item->title, oxine_main_menuitem_cb, item);
        otk_set_focus (b);
      }
      otk_draw_all (oxine->otk);
      return 1;
    case oxine_win_action_hide:
      oxine_main_close (oxine);
      return 1;
    case oxine_win_action_update:
     return 1;
    case oxine_win_action_delete:
      oxine_main_close (oxine);
      oxine->sessions[oxine_win_id_main] = NULL;
      while (session->menu_items.head.next->next) {
        menuitem_t *item;
        xitk_container (item, session->menu_items.head.next, node);
        xitk_dnode_remove (&item->node);
        if (item->title)
          ho_free (item->title);
        if (item->data)
          ho_free (item->data);
        ho_free (item);
      }
      ho_free (session);
      return 1;
    default: return 0;
  }
}

static void oxine_main_init (oxine_t *oxine) {
  oxine_main_session_t *session;
  if (!oxine)
    return;
  session = ho_new (oxine_main_session_t);
  if (!session)
    return;
  session->session.oxine = oxine;
  session->session.main = oxine_main_main;
  xitk_dlist_init (&session->menu_items);
  oxine->sessions[oxine_win_id_main] = &session->session;
}

#if 0
static void media_stop_cb(void *data);
static void media_info_cb(void *data);
static void media_pause_cb(void *data, int i);

//static void event_delay(void *data);

const char *playpause_strings[] = { "<", ">" };

static void media_pause_cb(void *data, int i) {
  oxine_t *oxine = (oxine_t*) data;

  switch(i) {
    case 1:
      odk_set_speed(oxine->odk, ODK_SPEED_NORMAL);
      break;
    case 2:
      odk_set_speed(oxine->odk, ODK_SPEED_PAUSE);
      break;
  }
}

static void format_time(char *buf, int sec) {
    sprintf(buf, "%d:%02d:%02d", sec/3600, (sec%3600)/60, ((sec%3600)%60));
}

static void media_stop_cb(void *data) {
  oxine_t *oxine = (oxine_t*) data;

  odk_stop(oxine->odk);
  oxine->main_menu_cb(oxine);
}

static void media_tracks_list_cb(void *data, void *entry) {
  oxine_t *oxine = (oxine_t *) data;
  char *mrl = (char *) entry;

  odk_stop(oxine->odk);
  odk_open_and_play(oxine->odk, mrl);
}

static void media_tracks_quit(void *data) {
  oxine_t *oxine = (oxine_t*) data;

  oxine->need_draw = 0;
  printf("%d\n", oxine->mode);
  oxine->mode = OXINE_MODE_NORMAL;
  printf("%d\n", oxine->mode);
  oxine->pauseplay = NULL;
  oxine_main_close (oxine);
}

static void media_tracks_cb(void *data) {
  oxine_t *oxine = (oxine_t *) data;
  int num, i = 0;
  char **str;
  otk_widget_t *list;

  oxine->pauseplay = NULL;
  oxine->main_window = NULL;
  oxine->info_window = NULL;
  oxine_main_close (oxine);
  oxine->main_window = otk_window_new (oxine->otk, NULL, 50, 50, 550, 500);
  list = otk_list_new(oxine->main_window, 10, 10, 500, 490, media_tracks_list_cb, oxine);

  str = xine_get_autoplay_mrls (oxine->xine, "CD", &num);
  while(i < num) {
    /* printf("%d %s\n", i, str[i]); */
    otk_add_listentry(list, str[i], str[i], -1);
    i++;
  }
  otk_list_set_pos(list, 0);
  otk_set_focus(list);
  otk_draw_all(oxine->otk);
}

static void info_button_time(void *data) {
  oxine_t *oxine = (oxine_t *)data;
  int ret, pos_time, length;

  ret = odk_get_pos_length_high(oxine->odk, NULL, &pos_time, &length);
  /* oxine->time = malloc(sizeof(char)*255); */
  if(ret) {
    pos_time /= 1000;
    length /= 1000;
    sprintf(oxine->time, "(%d:%02d:%02d / %d:%02d:%02d)", pos_time/3600, (pos_time%3600)/60, ((pos_time%3600)%60),
	    length/3600, (length%3600)/60, ((length%3600)%60));
  }
  else sprintf(oxine->time, "N/A");
}
#endif

#if 0
static void media_info_cb(void *data) {
  oxine_t *oxine = (oxine_t*) data;
  int pos_time, length, ret;
  int cline = 0;
  const char *buf1;
  char *buf2;
  /* otk_widget_t *b, *layout; */

  oxine->media_info_close_cb = media_info_close_cb;
  oxine->pauseplay = NULL;
  if (oxine->info_window) {
    otk_destroy_widget(oxine->info_window);
    oxine->info_window = NULL;
    otk_draw_all(oxine->otk);
    return;
  }
  /* oxine_main_close (oxine); */
  buf2 = NULL;
  buf1 = odk_get_mrl(oxine->odk);
  if (buf1) {
    buf2 = strrchr(buf1,'/');
    if (buf2) buf2++; else buf2 = buf1;
  }

  if (oxine->lines[0]) ho_free(oxine->lines[0]);
  if (oxine->lines[1]) ho_free(oxine->lines[1]);
  if (oxine->lines[2]) ho_free(oxine->lines[2]);

  oxine->lines[0] = odk_get_meta_info(oxine->odk, XINE_META_INFO_TITLE);
  if(!oxine->lines[0] && buf2) oxine->lines[0] = ho_strdup(buf2);
  if(oxine->lines[0]) cline++;
  buf1 = odk_get_meta_info(oxine->odk, XINE_META_INFO_ARTIST);
  buf2 = odk_get_meta_info(oxine->odk, XINE_META_INFO_ALBUM);
  if (buf1 && buf2) {
    oxine->lines[cline] = ho_newstring(strlen(buf1)+strlen(buf2)+10);
    sprintf(oxine->lines[cline], "%s: %s", buf1, buf2);
    cline++;
  } else if (buf1 && (strlen(buf1)>0)) {
    oxine->lines[cline++] = ho_strdup(buf1);
  } else if (buf2 && (strlen(buf2)>0)) {
    oxine->lines[cline++] = ho_strdup(buf2);
  }
  /*
  oxine->genre = odk_get_meta_info(oxine->odk, XINE_META_INFO_GENRE);
  if(!oxine->genre) oxine->genre = ho_strdup("Genre N/A");
  oxine->year = odk_get_meta_info(oxine->odk, XINE_META_INFO_YEAR);
  if(!oxine->year) oxine->year = ho_strdup("Year N/A");
*/
  ret = odk_get_pos_length_high(oxine->odk, NULL, &pos_time, &length);

  if(ret && (length > 0)) {
    oxine->lines[cline] = ho_newstring(255);
    pos_time /= 1000;
    length /= 1000;
    format_time(oxine->lines[cline], length);
  }

  oxine->info_window = otk_window_new (oxine->otk, NULL, 5, 5, 790, 240);

  if (oxine->lines[0])
  otk_label_new(oxine->info_window, 5, 40, OTK_ALIGN_LEFT|OTK_ALIGN_VCENTER, oxine->lines[0]);

  if (oxine->lines[1])
  otk_label_new(oxine->info_window, 5, 110, OTK_ALIGN_LEFT|OTK_ALIGN_VCENTER, oxine->lines[1]);

  if (oxine->lines[2])
  otk_label_new(oxine->info_window, 5, 180, OTK_ALIGN_LEFT|OTK_ALIGN_VCENTER, oxine->lines[2]);

  /*
  layout = otk_layout_new(oxine->info_window, 10, 10, 680, 480, 6, 1);

  b = otk_button_grid_new(oxine->title, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 0, 1, 1);
  otk_set_focus(b);

  b = otk_button_grid_new(oxine->artist, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 1, 1, 1);

  b = otk_button_grid_new(oxine->genre, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 2, 1, 1);

  b = otk_button_grid_new(oxine->album, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 3, 1, 1);

  b = otk_button_grid_new(oxine->year, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 4, 1, 1);

  b = otk_button_grid_new(oxine->time, media_freeandreturnto_cb, oxine);
  otk_layout_add_widget(layout, b, 0, 5, 1, 1);
  otk_button_uc_set(b, info_button_time, oxine);
 */

  otk_draw_all(oxine->otk);

  xitk_at_add (oxine->xitk, media_info_close_cb, oxine, 5000);
}
#endif

#if 0
static void playing_menu_update(void *data) {
  oxine_t *oxine = (oxine_t *) data;
  int pos_time, length;

  if (oxine->pos_str)
    if (odk_get_pos_length_high(oxine->odk, NULL, &pos_time, &length)) {
      pos_time /= 1000;
      format_time(oxine->pos_str, pos_time);
    }
}

static void playing_menu_cb(void *data) {
   oxine_t *oxine = (oxine_t *) data;
   otk_widget_t *b, *l;
   otk_widget_t *layout;
   int pos_time, length;

   oxine->pauseplay = NULL;
   if (oxine->main_window) {
     otk_destroy_widget(oxine->main_window);
     oxine->main_window = NULL;
     otk_draw_all(oxine->otk);
   }

   /* oxine_main_close (oxine); */

   if(oxine->mode == OXINE_MODE_PLAY_MENU) {
     oxine->mode = OXINE_MODE_NORMAL;
     return ;
   }
   oxine->mode = OXINE_MODE_PLAY_MENU;

   oxine->main_window = otk_window_new (oxine->otk, NULL, 50, 400, 700, 150);
   layout = otk_layout_new(oxine->main_window, 10, 10, 680, 130, 2, 6);
   oxine->pauseplay = otk_selector_grid_new (playpause_strings, 2, media_pause_cb, oxine);
   otk_layout_add_widget(layout, oxine->pauseplay, 0, 0, 1, 1);
   otk_set_focus(oxine->pauseplay);
   if (odk_get_speed(oxine->odk) == ODK_SPEED_PAUSE) otk_selector_set(oxine->pauseplay, 2);

   b = otk_button_grid_new ("}", media_stop_cb, oxine);
   otk_layout_add_widget(layout, b, 1, 0, 1, 1);

/*   b = otk_button_grid_new ("Volume", NULL, NULL);
   otk_layout_add_widget(layout, b, 0, 1, 1, 1);

   b = otk_slider_grid_new (odk_get_volume);
   otk_layout_add_widget(layout, b, 1, 1, 1, 1);

   if(!oxine->cd_in_use) {
     b = otk_button_grid_new("Seek", NULL, NULL);
     otk_layout_add_widget(layout, b, 0, 2, 1, 1);
   }else {
     b = otk_button_grid_new ("T", media_tracks_cb, oxine);
     otk_layout_add_widget(layout, b, 0, 2, 1, 1);
   }*/

   if(oxine->cd_in_use) {
     b = otk_button_grid_new ("T", media_tracks_cb, oxine);
     otk_layout_add_widget(layout, b, 3, 0, 1, 1);
   }

   b = otk_slider_grid_new (odk_get_seek);
   otk_layout_add_widget(layout, b, 2, 1, 4, 1);
   otk_set_update(b,1);

   b = otk_button_grid_new ("i", media_info_cb, oxine);
   otk_layout_add_widget(layout, b, 2, 0, 1, 1);

   if (!oxine->pos_str) oxine->pos_str = ho_newstring(64);
   if (odk_get_pos_length_high(oxine->odk, NULL, &pos_time, &length)) {
     pos_time /= 1000;
     format_time(oxine->pos_str, pos_time);
   }

   l = otk_label_new (oxine->main_window,  110, 100, OTK_ALIGN_CENTER|OTK_ALIGN_VCENTER, oxine->pos_str);
   otk_set_update(l,1);
   otk_button_uc_set(b, playing_menu_update, oxine);

   /* oxine->need_draw = 1; */

   otk_draw_all(oxine->otk);
}
#endif

#if 0
static void main_menu_cb(void *data) {
  oxine_t *oxine = (oxine_t*) data;

  oxine->main_window = otk_window_new (oxine->otk, NULL, 50, 130, 700, 420);

  /*
  b = otk_button_new (oxine->main_window, 50, 45, 290, 60, "Play Disc", disc_cb, oxine);
  otk_set_focus(b);
  */

  b = otk_button_new (oxine->main_window, 360, 45, 290, 60, "Mediamarks", mediamarks_cb, oxine);
  otk_set_focus(b);

  /*
-  b = otk_button_new (oxine->main_window, 50, 150, 290, 60, "Analogue TV", tv_cb, oxine);
  otk_set_focus(b);
  */

  otk_button_new (oxine->main_window, 360, 150, 290, 60, "Playlist", playlist_cb, oxine);

  /*
  otk_button_new (oxine->main_window, 50, 255, 290, 60, "Digital TV", dvb_cb, oxine);

  otk_button_new (oxine->main_window, 360, 255, 290, 60, "Control", control_cb, oxine);
  */

  otk_button_new (oxine->main_window, 205, 320, 290, 60, "Shutdown", shutdown_cb, oxine);

/*  otk_button_new (oxine->main_window, 50, 180, 290, 60, "File", file_cb, oxine);
  otk_button_new (oxine->main_window, 50, 260, 290, 60, "Streaming", streaming_cb, oxine);*/

  otk_draw_all(oxine->otk);
}
#endif

static void return_cb(void *this) {
  oxine_t *oxine = (oxine_t*) this;

  oxine_win_main (oxine, oxine->last_win, oxine_win_action_show);
}

static void oxine_error_msg (gGui_t *gui, const char *text) {
  oxine_t *oxine = gui->oxine;
  otk_widget_t *b;
  int l = strlen (text);
  char *text2, *s;

  (void)gui;
  s = text2 = malloc (sizeof (OTK_TEMP_STR) + l);
  if (!text2)
    return;
  s += sizeof (OTK_TEMP_STR) - 1;
  memcpy (s, text, l + 1);

#ifdef OXINE_LOCK
  oxine = oxine_instance_get (oxine);
  if (!oxine)
    return;
#endif

  oxine_main_close (oxine);
  oxine->main_window = otk_window_new (oxine->otk, NULL, 100, 150, 600, 300);

  for( l = 0; l < 4 && s && strlen(s); l++ ) {
    char *line = s;
    otk_widget_t *label;

    if( (s = strchr(s,'\n')) ) {
      *s++ = '\0';
    }

    memcpy (line - (sizeof (OTK_TEMP_STR) - 1), OTK_TEMP_STR, sizeof (OTK_TEMP_STR) - 1);
    label = otk_label_new (oxine->main_window, 300, 30 + l * 25,
      OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER, line - (sizeof (OTK_TEMP_STR) - 1));
    otk_label_set_font_size(label, 20);
  }

  b = otk_button_new(oxine->main_window, 260, 240, 80, 50, "OK", return_cb, oxine);
  otk_set_focus(b);
  otk_draw_all(oxine->otk);

  free(text2);

#ifdef OXINE_LOCK
  oxine_instance_unget(oxine);
#endif
}

/*
 * initialisation
 */
int oxine_is_visible (oxine_t *oxine) {
  return oxine ? !!oxine->main_window : 0;
}

void oxine_menu (oxine_t *oxine) {
#ifdef OXINE_LOCK
  oxine = oxine_instance_get (oxine);
#endif
  if (!oxine)
    return;
  oxine_adapt (oxine);
  if (!oxine->main_window) {
    video_window_reset_ssaver (oxine->gui->vwin, 1);
    oxine->gui->nongui_error_msg = oxine_error_msg;
    oxine_win_main (oxine, oxine->last_win, oxine_win_action_show);
  } else {
    if (!oxine_win_main (oxine, oxine->last_win, oxine_win_action_hide))
      otk_clear (oxine->otk);
  }

#ifdef OXINE_LOCK
  oxine_instance_unget(oxine);
#endif
}

int oxine_action_event (oxine_t *oxine, int xine_event_type) {
  oxine_event_t ev;

#ifdef OXINE_LOCK
  oxine = oxine_instance_get (oxine);
#endif
  if (!oxine)
    return 0;

  if (!oxine->main_window) {
#ifdef OXINE_LOCK
    oxine_instance_unget(oxine);
#endif
    return 0;
  }

  ev.type = OXINE_EVENT_KEY;

  switch( xine_event_type ) {
  default:
#ifdef OXINE_LOCK
    oxine_instance_unget(oxine);
#endif
    return 0;
  case XINE_EVENT_INPUT_UP:
    ev.key = OXINE_KEY_UP;
    break;
  case XINE_EVENT_INPUT_DOWN:
    ev.key = OXINE_KEY_DOWN;
    break;
  case XINE_EVENT_INPUT_LEFT:
    ev.key = OXINE_KEY_LEFT;
    break;
  case XINE_EVENT_INPUT_RIGHT:
    ev.key = OXINE_KEY_RIGHT;
    break;
  case XINE_EVENT_INPUT_SELECT:
    oxine->gui->nongui_error_msg = oxine_error_msg;
    ev.key = OXINE_KEY_SELECT;
    break;
  }

  video_window_reset_ssaver (oxine->gui->vwin, 1);
  otk_send_event(oxine->otk, &ev);
#ifdef OXINE_LOCK
  oxine_instance_unget(oxine);
#endif
  return 1;
}

int oxine_event (oxine_t *oxine, const oxine_event_t *e) {
  oxine_event_t e2;
  int r;

#ifdef OXINE_LOCK
  oxine = oxine_instance_get (oxine);
#endif
  if (!oxine || !e)
    return 0;
  if (!oxine->main_window) {
#ifdef OXINE_LOCK
    oxine_instance_unget (oxine);
#endif
    return 0;
  }
  e2 = *e;
  r = otk_send_event (oxine->otk, &e2);
#ifdef OXINE_LOCK
  oxine_instance_unget (oxine);
#endif
  if (oxine->shutdown)
    gui_action_id (oxine->gui, ACTID_QUIT);
  return r;
}

void oxine_adapt (oxine_t *oxine) {
  oxine_event_t ev;

#ifdef OXINE_LOCK
  oxine = oxine_instance_get (oxine);
#endif
  if (!oxine)
    return;

  ev.type = OXINE_EVENT_FORMAT_CHANGED;
  otk_send_event(oxine->otk, &ev);
#ifdef OXINE_LOCK
  oxine_instance_unget(oxine);
#endif
}

oxine_t *oxine_init (gGui_t *gui) {
  oxine_t *oxine;
  xine_cfg_entry_t centry;
  int i;

  if (!gui)
    return NULL;
  oxine = calloc (1, sizeof (*oxine));
  if (!oxine)
    return NULL;
#ifdef OXINE_LOCK
  pthread_mutex_init (&oxine->instance.mutex, NULL);
  pthread_cond_init (&oxine->instance.unlocked, NULL);
  oxine->instance.locks = 0;
  pthread_mutex_lock (&oxine->instance.mutex);
#endif

  oxine->gui = gui;
  oxine->xine = gui->xine;
  oxine->xitk = gui->xitk;
  oxine->main_window = NULL;
  for (i = 0; i < oxine_win_id_LAST; i++)
    oxine->sessions[i] = NULL;
  oxine_main_init (oxine);
  oxine_playlist_init (oxine);
  oxine_mediamarks_init (oxine);
  oxine->last_win = oxine_win_id_main;

  oxine->cd_mountpoint = xine_config_register_filename (oxine->xine, "gui.osdmenu.dvd_mountpoint",
    "/dvd", XINE_CONFIG_STRING_IS_DIRECTORY_NAME,
    "directory a media in dvd device will be mounted",
    "directory a media in dvd device will be mounted",
    10, NULL, NULL);
  if (xine_config_lookup_entry (oxine->xine, "input.dvd_device", &centry)) {
    oxine->cd_device = centry.str_value;
  }

  oxine->odk = odk_init (gui);
  oxine->otk = otk_init (oxine, oxine->gui->verbosity);

#ifdef OXINE_LOCK
  oxine->instance.locks = 1;
  pthread_mutex_unlock (&oxine->instance.mutex);
#endif
  return oxine;
}

void oxine_exit (oxine_t **_oxine) {
  if (_oxine && *_oxine) {
    oxine_t *oxine = *_oxine;
    int i;

    *_oxine = NULL;
#ifdef OXINE_LOCK
    pthread_mutex_lock (&oxine->instance.mutex);
    oxine->instance.locks--;
    while (oxine->instance.locks > 0)
      pthread_cond_wait (&oxine->instance.unlocked, &oxine->instance.mutex);
#endif
    for (i = 0; i < oxine_win_id_LAST; i++)
      if (oxine->sessions[i])
        oxine->sessions[i]->main (oxine, i, oxine_win_action_delete);
    if (oxine->otk)
      otk_free (oxine->otk);
    if (oxine->odk)
      odk_free (oxine->odk);
#ifdef DEBUG
    heapstat ();
#endif
#ifdef OXINE_LOCK
    pthread_mutex_unlock (&oxine->instance.mutex);
    pthread_cond_destroy (&oxine->instance.unlocked);
    pthread_mutex_destroy (&oxine->instance.mutex);
#endif
    free (oxine);
  }
}

int oxine_win_main (oxine_t *oxine, oxine_win_id_t id, oxine_win_action_t action) {
  if (!XITK_0_TO_MAX_MINUS_1 (id, oxine_win_action_LAST))
    return 0;
  if (oxine && oxine->sessions[id]) {
    if (action == oxine_win_action_show) {
      if (oxine->sessions[oxine->last_win])
        oxine->sessions[oxine->last_win]->main (oxine, oxine->last_win, oxine_win_action_hide);
      oxine->last_win = id;
    }
    return oxine->sessions[id]->main (oxine, id, action);
  }
  return 0;
}
