# frozen_string_literal: true

require "action_view/template/resolver"

module ActionView #:nodoc:
  # Use FixtureResolver in your tests to simulate the presence of files on the
  # file system. This is used internally by Rails' own test suite, and is
  # useful for testing extensions that have no way of knowing what the file
  # system will look like at runtime.
  class FixtureResolver < OptimizedFileSystemResolver
    def initialize(hash = {}, pattern = nil)
      super("")
      if pattern
        ActiveSupport::Deprecation.warn "Specifying a custom path for #{self.class} is deprecated. Implement a custom Resolver subclass instead."
        @pattern = pattern
      end
      @hash = hash
      @path = ""
    end

    def data
      @hash
    end

    def to_s
      @hash.keys.join(", ")
    end

    private
      def query(path, exts, _, locals, cache:)
        regex = build_regex(path, exts)

        @hash.select do |_path, _|
          ("/" + _path).match?(regex)
        end.map do |_path, source|
          handler, format, variant = extract_handler_and_format_and_variant(_path)

          Template.new(source, _path, handler,
            virtual_path: path.virtual,
            format: format,
            variant: variant,
            locals: locals
          )
        end.sort_by do |t|
          match = ("/" + t.identifier).match(regex)
          EXTENSIONS.keys.reverse.map do |ext|
            if ext == :variants && exts[ext] == :any
              match[ext].nil? ? 0 : 1
            elsif match[ext].nil?
              exts[ext].length
            else
              found = match[ext].to_sym
              exts[ext].index(found)
            end
          end
        end
      end
  end

  class NullResolver < PathResolver
    def query(path, exts, _, locals, cache:)
      handler, format, variant = extract_handler_and_format_and_variant(path)
      [ActionView::Template.new("Template generated by Null Resolver", path.virtual, handler, virtual_path: path.virtual, format: format, variant: variant, locals: locals)]
    end
  end
end
