//////////////////////////////////////////////////////////////////////
// Score.h
//
// Score scalar data type.
//////////////////////////////////////////////////////////////////////

#ifndef SCORE_H
#define SCORE_H

#include <stdio.h>
#include <math.h>
#include "Assert.h"


//////////////////////////////////////////////////////////////////////
// SCORE datatype math constants
//////////////////////////////////////////////////////////////////////

typedef int SCORE;

const int TABLE_SIZE = 32768;
const float TABLE_UPPER_LIMIT = 8.0;
const float SCALE = (float) TABLE_SIZE / TABLE_UPPER_LIMIT;

const SCORE ZERO_SCORE = 0;
const SCORE ONE_SCORE = (SCORE) SCALE;
const SCORE LOG_ZERO_SCORE = -1000000000;
const SCORE LOG_ONE_SCORE = 0;

extern SCORE *LOG_EXP_PLUS_1_TABLE;
extern float *EXP_SCORE_TO_FLOAT_TABLE;

//////////////////////////////////////////////////////////////////////
// Floating point math constants
//////////////////////////////////////////////////////////////////////

const float ZERO_FLOAT = 0.0f;
const float ONE_FLOAT = 1.0f;
const float LOG_ZERO_FLOAT = -2e20f;
const float LOG_ONE_FLOAT = 0.0f;

// precompute math tables
void PRECOMPUTE_SCORE_TABLES();

//////////////////////////////////////////////////////////////////////
// Convert float to score
//////////////////////////////////////////////////////////////////////

inline SCORE TO_SCORE (float score){
  return (SCORE) (score * SCALE);
}

//////////////////////////////////////////////////////////////////////
// Convert score to float
//////////////////////////////////////////////////////////////////////

inline float TO_FLOAT (const SCORE &score){
  return (float) score / SCALE;
}

//////////////////////////////////////////////////////////////////////
// Compute log (x)
//////////////////////////////////////////////////////////////////////

inline float LOG_FLOAT (float x){
#ifdef EXACT
  return log(x);
#else
  float value;

  if (x > 1) return log (x);
  value = 0;

  while (x < 0.5f){
    x += x;
    value += -0.6931471806f;
  }
  
  return value + (((-0.8921177528f*x + 3.5313113007f)*x - 5.8206844725f)*x + 5.6098099262f)*x - 2.4284166653f;
#endif
}

//////////////////////////////////////////////////////////////////////
// Compute exp (x) / SCALE
//////////////////////////////////////////////////////////////////////

inline float EXP_SCORE_TO_FLOAT (SCORE x){
  if (x >= 0) return 1.0f;
  if (-x >= TABLE_SIZE) return 0.0f;
  return EXP_SCORE_TO_FLOAT_TABLE[-x];
}

//////////////////////////////////////////////////////////////////////
// Compute exp (x)
//////////////////////////////////////////////////////////////////////

inline float EXP_FLOAT (float x){

#ifdef EXACT
  return exp(x);
#else
  if (x > -2){
    if (x > -0.5){
      if (x > 0)
	return exp(x);
      return (((0.03254409303190190000*x + 0.16280432765779600000)*x + 0.49929760485974900000)*x + 0.99995149601363700000)*x + 0.99999925508501600000;
    }
    if (x > -1)
      return (((0.01973899026052090000*x + 0.13822379685007000000)*x + 0.48056651562365000000)*x + 0.99326940370383500000)*x + 0.99906756856399500000;
    return (((0.00940528203591384000*x + 0.09414963667859410000)*x + 0.40825793595877300000)*x + 0.93933625499130400000)*x + 0.98369508190545300000;
  }
  if (x > -8){
    if (x > -4)
      return (((0.00217245711583303000*x + 0.03484829428350620000)*x + 0.22118199801337800000)*x + 0.67049462206469500000)*x + 0.83556950223398500000;
    return (((0.00012398771025456900*x + 0.00349155785951272000)*x + 0.03727721426017900000)*x + 0.17974997741536900000)*x + 0.33249299994217400000;
  }
  if (x > -16)
    return (((0.00000051741713416603*x + 0.00002721456879608080)*x + 0.00053418601865636800)*x + 0.00464101989351936000)*x + 0.01507447981459420000;
  return 0;
  #endif
}

//////////////////////////////////////////////////////////////////////
// Computes log (exp (x) + 1)
//////////////////////////////////////////////////////////////////////

inline float LOOKUP_FLOAT (float x){
#ifdef EXACT
  return log(exp(x)+1);
#else
  if (x < 2){
    if (x < 0.5){
      if (x < 0)
	return log (exp(x) + 1);
      return (((-0.00486373205785640000*x - 0.00020245408813934800)*x + 0.12504222666029800000)*x + 0.49999685320563000000)*x + 0.69314723138948900000;
    }
    if (x < 1)
      return (((-0.00278634205460548000*x - 0.00458097251248546000)*x + 0.12865849880472500000)*x + 0.49862228499205200000)*x + 0.69334810088688000000;
    return (((0.00059633755154209200*x - 0.01918996666063320000)*x + 0.15288232492093800000)*x + 0.48039958825756900000)*x + 0.69857578503189200000;
  }
  if (x < 8){
    if (x < 4)
      return (((0.00135958539181047000*x - 0.02329807659316430000)*x + 0.15885799609532100000)*x + 0.48167498563270800000)*x + 0.69276185058669200000;
    return (((0.00011992394456683500*x - 0.00338464503306568000)*x + 0.03622746366545470000)*x + 0.82481250248383700000)*x + 0.32507892994863100000;
  }
  if (x < 16)
    return (((0.00000051726300753785*x - 0.00002720671238876090)*x + 0.00053403733818413500)*x + 0.99536021775747900000)*x + 0.01507065715532010000;
  return x;
#endif
}

//////////////////////////////////////////////////////////////////////
// Computes sum of two numbers in log space
//////////////////////////////////////////////////////////////////////

inline float LOG_ADD_FLOAT (float x, float y){
  if (x < y){ float t = x; x = y; y = t; }
  if (y <= LOG_ZERO_FLOAT) return x;
  return LOOKUP_FLOAT(x-y) + y;
}

//////////////////////////////////////////////////////////////////////
// Computes sum of two numbers in log space
//////////////////////////////////////////////////////////////////////

inline SCORE LOG_ADD_SCORE (SCORE x, SCORE y){
  if (x <= y){
    if (y - x >= TABLE_SIZE) return y;
    return LOG_EXP_PLUS_1_TABLE[y - x] + x;
  }

  if (x - y >= TABLE_SIZE) return x;
  return LOG_EXP_PLUS_1_TABLE[x - y] + y;
}

#endif

