/*************************************************************************
 * Copyright (C) 2019-2024 Intel Corporation
 *
 * Licensed under the Apache License,  Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * 	http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law  or agreed  to  in  writing,  software
 * distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the  specific  language  governing  permissions  and
 * limitations under the License.
 *************************************************************************/

/*
For squaring, an optimized approach is utilized. As an example, suppose we are multiplying two 4-digit numbers:
                                    | a3 | a2 | a1 | a0 |
                                    | b3 | b2 | b1 | b0 |
                                  X______________________

                | a3 * b3 | a2 * b2 | a1 * b1 | a0 * b0 |
                     | a3 * b2 | a2 * b1 | a1 * b0 |
                     | a2 * b3 | a1 * b2 | a0 * b1 |
                          | a3 * b1 | a2 * b0 |
                          | a1 * b3 | a0 * b2 |
                               | a3 * b0 |
                               | a0 * b3 |

This operation is realized with 4x4=16 digit-wise multiplications. When a=b (for squaring), multiplication operation is optimizes as follows:
                                    | a3 | a2 | a1 | a0 |
                                    | a3 | a2 | a1 | a0 |
                                  X______________________

                | a3 * a3 | a2 * a2 | a1 * a1 | a0 * a0 |
            2*       | a3 * a2 | a2 * a1 | a1 * a0 |

            2*            | a3 * a1 | a2 * a0 |

            2*                 | a3 * a0 |

This operation is realized with 10 digit-wise multiplications. For an n-digit squaring operation, (n^2-n)/2 less digit-wise multiplications are
required. The number of digit-wise multiplications for n-digit squaring can be calculated with the following equation:

    n^2 - (n^2-n)/2

multiplication by 2 operations are realized with add64 instructions.
*/

#include <internal/common/ifma_defs.h>

#if ((_MBX == _MBX_L9) && _MBX_AVX_IFMA_SUPPORTED)

#include <internal/rsa/avxifma_ams.h>

static void ams52x60_square_diagonal_mb4(__m256i* res, const int64u* inpA_mb)
{
    const __m256i* inpA = (const __m256i*)inpA_mb;
    register __m256i r0, r1, r2, r3, r4, r5, r6, r7, r8, AL;
    const int N             = 60;
    const __m256i zero_simd = _mm256_setzero_si256();

    // 1st triangle - sum the products, double and square
    r0 = zero_simd;

    res[0]  = _mm256_madd52lo_epu64(r0, inpA[0], inpA[0]);
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[1]);      // Sum(1)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[2]);      // Sum(2)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[3]);      // Sum(3)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[4]);      // Sum(4)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[5]);      // Sum(5)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[6]);      // Sum(6)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[7]);      // Sum(7)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[8]);      // Sum(8)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[1]);      // Sum(1)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[2]);      // Sum(2)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[3]);      // Sum(3)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[4]);      // Sum(4)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[5]);      // Sum(5)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[6]);      // Sum(6)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[7]);      // Sum(7)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[8]);      // Sum(8)
    AL      = inpA[1];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[2]);      // Sum(3)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[3]);      // Sum(4)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[4]);      // Sum(5)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[5]);      // Sum(6)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[6]);      // Sum(7)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[7]);      // Sum(8)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[2]);      // Sum(3)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[3]);      // Sum(4)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[4]);      // Sum(5)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[5]);      // Sum(6)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[6]);      // Sum(7)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[7]);      // Sum(8)
    AL      = inpA[2];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[3]);      // Sum(5)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[4]);      // Sum(6)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[5]);      // Sum(7)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[6]);      // Sum(8)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[3]);      // Sum(5)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[4]);      // Sum(6)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[5]);      // Sum(7)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[6]);      // Sum(8)
    AL      = inpA[3];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[4]);      // Sum(7)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[5]);      // Sum(8)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[4]);      // Sum(7)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[5]);      // Sum(8)
    r0      = _mm256_add_epi64(r0, r0);                    // Double(1)
    r0      = _mm256_madd52hi_epu64(r0, inpA[0], inpA[0]); // Add square(1)
    res[1]  = r0;
    r1      = _mm256_add_epi64(r1, r1);                    // Double(2)
    r1      = _mm256_madd52lo_epu64(r1, inpA[1], inpA[1]); // Add square(2)
    res[2]  = r1;
    r2      = _mm256_add_epi64(r2, r2);                    // Double(3)
    r2      = _mm256_madd52hi_epu64(r2, inpA[1], inpA[1]); // Add square(3)
    res[3]  = r2;
    r3      = _mm256_add_epi64(r3, r3);                    // Double(4)
    r3      = _mm256_madd52lo_epu64(r3, inpA[2], inpA[2]); // Add square(4)
    res[4]  = r3;
    r4      = _mm256_add_epi64(r4, r4);                    // Double(5)
    r4      = _mm256_madd52hi_epu64(r4, inpA[2], inpA[2]); // Add square(5)
    res[5]  = r4;
    r5      = _mm256_add_epi64(r5, r5);                    // Double(6)
    r5      = _mm256_madd52lo_epu64(r5, inpA[3], inpA[3]); // Add square(6)
    res[6]  = r5;
    r6      = _mm256_add_epi64(r6, r6);                    // Double(7)
    r6      = _mm256_madd52hi_epu64(r6, inpA[3], inpA[3]); // Add square(7)
    res[7]  = r6;
    r7      = _mm256_add_epi64(r7, r7);                    // Double(8)
    r7      = _mm256_madd52lo_epu64(r7, inpA[4], inpA[4]); // Add square(8)
    res[8]  = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[9]);      // Sum(9)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[10]);     // Sum(10)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[11]);     // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[12]);     // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[13]);     // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[14]);     // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[15]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[16]);     // Sum(16)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[9]);      // Sum(9)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[10]);     // Sum(10)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[11]);     // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[12]);     // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[13]);     // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[14]);     // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[15]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[16]);     // Sum(16)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[8]);      // Sum(9)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[9]);      // Sum(10)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[10]);     // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[11]);     // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[12]);     // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[13]);     // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[14]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[15]);     // Sum(16)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[8]);      // Sum(9)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[9]);      // Sum(10)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[10]);     // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[11]);     // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[12]);     // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[13]);     // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[14]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[15]);     // Sum(16)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[7]);      // Sum(9)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[8]);      // Sum(10)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[9]);      // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[10]);     // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[11]);     // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[12]);     // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[13]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[14]);     // Sum(16)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[7]);      // Sum(9)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[8]);      // Sum(10)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[9]);      // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[10]);     // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[11]);     // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[12]);     // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[13]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[14]);     // Sum(16)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[6]);      // Sum(9)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[7]);      // Sum(10)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[8]);      // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[9]);      // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[10]);     // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[11]);     // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[12]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[13]);     // Sum(16)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[6]);      // Sum(9)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[7]);      // Sum(10)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[8]);      // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[9]);      // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[10]);     // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[11]);     // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[12]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[13]);     // Sum(16)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[5]);      // Sum(9)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[6]);      // Sum(10)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[7]);      // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[8]);      // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[9]);      // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[10]);     // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[11]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[12]);     // Sum(16)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[5]);      // Sum(9)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[6]);      // Sum(10)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[7]);      // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[8]);      // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[9]);      // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[10]);     // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[11]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[12]);     // Sum(16)
    AL      = inpA[5];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[6]);      // Sum(11)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[7]);      // Sum(12)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[8]);      // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[9]);      // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[10]);     // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[11]);     // Sum(16)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[6]);      // Sum(11)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[7]);      // Sum(12)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[8]);      // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[9]);      // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[10]);     // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[11]);     // Sum(16)
    AL      = inpA[6];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[7]);      // Sum(13)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[8]);      // Sum(14)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[9]);      // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[10]);     // Sum(16)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[7]);      // Sum(13)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[8]);      // Sum(14)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[9]);      // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[10]);     // Sum(16)
    AL      = inpA[7];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[8]);      // Sum(15)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[9]);      // Sum(16)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[8]);      // Sum(15)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[9]);      // Sum(16)
    r0      = _mm256_add_epi64(r0, r0);                    // Double(9)
    r0      = _mm256_madd52hi_epu64(r0, inpA[4], inpA[4]); // Add square(9)
    res[9]  = r0;
    r1      = _mm256_add_epi64(r1, r1);                    // Double(10)
    r1      = _mm256_madd52lo_epu64(r1, inpA[5], inpA[5]); // Add square(10)
    res[10] = r1;
    r2      = _mm256_add_epi64(r2, r2);                    // Double(11)
    r2      = _mm256_madd52hi_epu64(r2, inpA[5], inpA[5]); // Add square(11)
    res[11] = r2;
    r3      = _mm256_add_epi64(r3, r3);                    // Double(12)
    r3      = _mm256_madd52lo_epu64(r3, inpA[6], inpA[6]); // Add square(12)
    res[12] = r3;
    r4      = _mm256_add_epi64(r4, r4);                    // Double(13)
    r4      = _mm256_madd52hi_epu64(r4, inpA[6], inpA[6]); // Add square(13)
    res[13] = r4;
    r5      = _mm256_add_epi64(r5, r5);                    // Double(14)
    r5      = _mm256_madd52lo_epu64(r5, inpA[7], inpA[7]); // Add square(14)
    res[14] = r5;
    r6      = _mm256_add_epi64(r6, r6);                    // Double(15)
    r6      = _mm256_madd52hi_epu64(r6, inpA[7], inpA[7]); // Add square(15)
    res[15] = r6;
    r7      = _mm256_add_epi64(r7, r7);                    // Double(16)
    r7      = _mm256_madd52lo_epu64(r7, inpA[8], inpA[8]); // Add square(16)
    res[16] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(24)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[16]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[17]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[18]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[19]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[20]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[21]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[22]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[23]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[16]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[17]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[18]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[19]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[20]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[21]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[22]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[23]);       // Sum(24)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[15]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[16]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[17]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[18]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[19]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[20]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[21]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[22]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[15]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[16]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[17]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[18]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[19]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[20]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[21]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[22]);       // Sum(24)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[14]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[15]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[16]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[17]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[18]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[19]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[20]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[21]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[14]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[15]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[16]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[17]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[18]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[19]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[20]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[21]);       // Sum(24)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[13]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[14]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[15]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[16]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[17]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[18]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[19]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[20]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[13]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[14]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[15]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[16]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[17]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[18]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[19]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[20]);       // Sum(24)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[12]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[13]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[14]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[15]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[16]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[17]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[18]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[19]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[12]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[13]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[14]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[15]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[16]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[17]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[18]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[19]);       // Sum(24)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[11]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[12]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[13]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[14]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[15]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[16]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[17]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[18]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[11]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[12]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[13]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[14]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[15]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[16]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[17]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[18]);       // Sum(24)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[10]);       // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[11]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[12]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[13]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[14]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[15]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[16]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[17]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[10]);       // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[11]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[12]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[13]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[14]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[15]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[16]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[17]);       // Sum(24)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[9]);        // Sum(17)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[10]);       // Sum(18)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[11]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[12]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[13]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[14]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[15]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[16]);       // Sum(24)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[9]);        // Sum(17)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[10]);       // Sum(18)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[11]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[12]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[13]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[14]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[15]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[16]);       // Sum(24)
    AL      = inpA[9];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[10]);       // Sum(19)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[11]);       // Sum(20)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[12]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[13]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[14]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[15]);       // Sum(24)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[10]);       // Sum(19)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[11]);       // Sum(20)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[12]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[13]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[14]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[15]);       // Sum(24)
    AL      = inpA[10];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[11]);       // Sum(21)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[12]);       // Sum(22)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[13]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[14]);       // Sum(24)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[11]);       // Sum(21)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[12]);       // Sum(22)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[13]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[14]);       // Sum(24)
    AL      = inpA[11];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[12]);       // Sum(23)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[13]);       // Sum(24)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[12]);       // Sum(23)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[13]);       // Sum(24)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(17)
    r0      = _mm256_madd52hi_epu64(r0, inpA[8], inpA[8]);   // Add square(17)
    res[17] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(18)
    r1      = _mm256_madd52lo_epu64(r1, inpA[9], inpA[9]);   // Add square(18)
    res[18] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(19)
    r2      = _mm256_madd52hi_epu64(r2, inpA[9], inpA[9]);   // Add square(19)
    res[19] = r2;
    r3      = _mm256_add_epi64(r3, r3);                      // Double(20)
    r3      = _mm256_madd52lo_epu64(r3, inpA[10], inpA[10]); // Add square(20)
    res[20] = r3;
    r4      = _mm256_add_epi64(r4, r4);                      // Double(21)
    r4      = _mm256_madd52hi_epu64(r4, inpA[10], inpA[10]); // Add square(21)
    res[21] = r4;
    r5      = _mm256_add_epi64(r5, r5);                      // Double(22)
    r5      = _mm256_madd52lo_epu64(r5, inpA[11], inpA[11]); // Add square(22)
    res[22] = r5;
    r6      = _mm256_add_epi64(r6, r6);                      // Double(23)
    r6      = _mm256_madd52hi_epu64(r6, inpA[11], inpA[11]); // Add square(23)
    res[23] = r6;
    r7      = _mm256_add_epi64(r7, r7);                      // Double(24)
    r7      = _mm256_madd52lo_epu64(r7, inpA[12], inpA[12]); // Add square(24)
    res[24] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(32)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(32)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(32)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(32)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(32)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[20]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[21]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[20]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[21]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(32)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[19]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[20]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[21]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[22]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[19]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[20]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[21]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[22]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(32)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[18]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[19]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[20]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[21]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[22]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[23]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[18]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[19]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[20]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[21]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[22]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[23]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(32)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(32)
    AL      = inpA[9];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[16]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[17]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[18]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[19]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[20]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[21]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[22]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[23]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[16]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[17]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[18]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[19]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[20]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[21]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[22]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[23]);       // Sum(32)
    AL      = inpA[10];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[15]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[16]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[17]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[18]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[19]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[20]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[21]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[22]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[15]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[16]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[17]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[18]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[19]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[20]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[21]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[22]);       // Sum(32)
    AL      = inpA[11];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[14]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[15]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[16]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[17]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[18]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[19]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[20]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[21]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[14]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[15]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[16]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[17]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[18]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[19]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[20]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[21]);       // Sum(32)
    AL      = inpA[12];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[13]);       // Sum(25)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[14]);       // Sum(26)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[15]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[16]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[17]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[18]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[19]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[20]);       // Sum(32)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[13]);       // Sum(25)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[14]);       // Sum(26)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[15]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[16]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[17]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[18]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[19]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[20]);       // Sum(32)
    AL      = inpA[13];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[14]);       // Sum(27)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[15]);       // Sum(28)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[16]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[17]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[18]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[19]);       // Sum(32)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[14]);       // Sum(27)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[15]);       // Sum(28)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[16]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[17]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[18]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[19]);       // Sum(32)
    AL      = inpA[14];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[15]);       // Sum(29)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[16]);       // Sum(30)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[17]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[18]);       // Sum(32)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[15]);       // Sum(29)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[16]);       // Sum(30)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[17]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[18]);       // Sum(32)
    AL      = inpA[15];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[16]);       // Sum(31)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[17]);       // Sum(32)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[16]);       // Sum(31)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[17]);       // Sum(32)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(25)
    r0      = _mm256_madd52hi_epu64(r0, inpA[12], inpA[12]); // Add square(25)
    res[25] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(26)
    r1      = _mm256_madd52lo_epu64(r1, inpA[13], inpA[13]); // Add square(26)
    res[26] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(27)
    r2      = _mm256_madd52hi_epu64(r2, inpA[13], inpA[13]); // Add square(27)
    res[27] = r2;
    r3      = _mm256_add_epi64(r3, r3);                      // Double(28)
    r3      = _mm256_madd52lo_epu64(r3, inpA[14], inpA[14]); // Add square(28)
    res[28] = r3;
    r4      = _mm256_add_epi64(r4, r4);                      // Double(29)
    r4      = _mm256_madd52hi_epu64(r4, inpA[14], inpA[14]); // Add square(29)
    res[29] = r4;
    r5      = _mm256_add_epi64(r5, r5);                      // Double(30)
    r5      = _mm256_madd52lo_epu64(r5, inpA[15], inpA[15]); // Add square(30)
    res[30] = r5;
    r6      = _mm256_add_epi64(r6, r6);                      // Double(31)
    r6      = _mm256_madd52hi_epu64(r6, inpA[15], inpA[15]); // Add square(31)
    res[31] = r6;
    r7      = _mm256_add_epi64(r7, r7);                      // Double(32)
    r7      = _mm256_madd52lo_epu64(r7, inpA[16], inpA[16]); // Add square(32)
    res[32] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(40)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[36]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[37]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[38]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[39]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[36]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[37]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[38]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[39]);       // Sum(40)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[37]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[38]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[37]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[38]);       // Sum(40)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(40)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(40)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(40)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(40)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(40)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(40)
    AL      = inpA[9];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(40)
    AL      = inpA[10];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(40)
    AL      = inpA[11];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(40)
    AL      = inpA[12];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(40)
    AL      = inpA[13];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[20]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[21]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[20]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[21]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(40)
    AL      = inpA[14];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[19]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[20]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[21]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[22]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[19]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[20]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[21]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[22]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(40)
    AL      = inpA[15];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[18]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[19]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[20]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[21]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[22]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[23]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[18]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[19]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[20]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[21]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[22]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[23]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(40)
    AL      = inpA[16];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(33)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(34)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(40)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(33)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(34)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(40)
    AL      = inpA[17];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[18]);       // Sum(35)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[19]);       // Sum(36)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[20]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[21]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[22]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[23]);       // Sum(40)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[18]);       // Sum(35)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[19]);       // Sum(36)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[20]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[21]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[22]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[23]);       // Sum(40)
    AL      = inpA[18];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[19]);       // Sum(37)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[20]);       // Sum(38)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[21]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[22]);       // Sum(40)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[19]);       // Sum(37)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[20]);       // Sum(38)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[21]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[22]);       // Sum(40)
    AL      = inpA[19];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[20]);       // Sum(39)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[21]);       // Sum(40)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[20]);       // Sum(39)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[21]);       // Sum(40)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(33)
    r0      = _mm256_madd52hi_epu64(r0, inpA[16], inpA[16]); // Add square(33)
    res[33] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(34)
    r1      = _mm256_madd52lo_epu64(r1, inpA[17], inpA[17]); // Add square(34)
    res[34] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(35)
    r2      = _mm256_madd52hi_epu64(r2, inpA[17], inpA[17]); // Add square(35)
    res[35] = r2;
    r3      = _mm256_add_epi64(r3, r3);                      // Double(36)
    r3      = _mm256_madd52lo_epu64(r3, inpA[18], inpA[18]); // Add square(36)
    res[36] = r3;
    r4      = _mm256_add_epi64(r4, r4);                      // Double(37)
    r4      = _mm256_madd52hi_epu64(r4, inpA[18], inpA[18]); // Add square(37)
    res[37] = r4;
    r5      = _mm256_add_epi64(r5, r5);                      // Double(38)
    r5      = _mm256_madd52lo_epu64(r5, inpA[19], inpA[19]); // Add square(38)
    res[38] = r5;
    r6      = _mm256_add_epi64(r6, r6);                      // Double(39)
    r6      = _mm256_madd52hi_epu64(r6, inpA[19], inpA[19]); // Add square(39)
    res[39] = r6;
    r7      = _mm256_add_epi64(r7, r7);                      // Double(40)
    r7      = _mm256_madd52lo_epu64(r7, inpA[20], inpA[20]); // Add square(40)
    res[40] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[43]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[44]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[45]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[46]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[47]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[48]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[43]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[44]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[45]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[46]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[47]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[48]);       // Sum(48)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[40]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[41]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[42]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[43]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[44]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[45]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[46]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[47]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[40]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[41]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[42]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[43]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[44]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[45]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[46]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[47]);       // Sum(48)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[39]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[40]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[41]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[42]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[43]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[44]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[45]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[46]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[39]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[40]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[41]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[42]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[43]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[44]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[45]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[46]);       // Sum(48)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[38]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[39]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[40]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[41]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[42]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[43]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[44]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[45]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[38]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[39]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[40]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[41]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[42]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[43]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[44]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[45]);       // Sum(48)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[39]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[40]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[41]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[42]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[43]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[44]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[39]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[40]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[41]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[42]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[43]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[44]);       // Sum(48)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[39]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[40]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[41]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[42]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[43]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[39]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[40]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[41]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[42]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[43]);       // Sum(48)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[38]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[39]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[40]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[41]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[42]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[38]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[39]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[40]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[41]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[42]);       // Sum(48)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[37]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[38]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[39]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[40]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[41]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[37]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[38]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[39]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[40]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[41]);       // Sum(48)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(48)
    AL      = inpA[9];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[36]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[37]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[38]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[39]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[36]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[37]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[38]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[39]);       // Sum(48)
    AL      = inpA[10];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[37]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[38]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[37]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[38]);       // Sum(48)
    AL      = inpA[11];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(48)
    AL      = inpA[12];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(48)
    AL      = inpA[13];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(48)
    AL      = inpA[14];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(48)
    AL      = inpA[15];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(48)
    AL      = inpA[16];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(48)
    AL      = inpA[17];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(48)
    AL      = inpA[18];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(48)
    AL      = inpA[19];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(48)
    AL      = inpA[20];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(41)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(42)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(48)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(41)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(42)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(48)
    AL      = inpA[21];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(43)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(44)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(48)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(43)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(44)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(48)
    AL      = inpA[22];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(45)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(46)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(48)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(45)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(46)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(48)
    AL      = inpA[23];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(47)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(48)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(47)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(48)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(41)
    r0      = _mm256_madd52hi_epu64(r0, inpA[20], inpA[20]); // Add square(41)
    res[41] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(42)
    r1      = _mm256_madd52lo_epu64(r1, inpA[21], inpA[21]); // Add square(42)
    res[42] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(43)
    r2      = _mm256_madd52hi_epu64(r2, inpA[21], inpA[21]); // Add square(43)
    res[43] = r2;
    r3      = _mm256_add_epi64(r3, r3);                      // Double(44)
    r3      = _mm256_madd52lo_epu64(r3, inpA[22], inpA[22]); // Add square(44)
    res[44] = r3;
    r4      = _mm256_add_epi64(r4, r4);                      // Double(45)
    r4      = _mm256_madd52hi_epu64(r4, inpA[22], inpA[22]); // Add square(45)
    res[45] = r4;
    r5      = _mm256_add_epi64(r5, r5);                      // Double(46)
    r5      = _mm256_madd52lo_epu64(r5, inpA[23], inpA[23]); // Add square(46)
    res[46] = r5;
    r6      = _mm256_add_epi64(r6, r6);                      // Double(47)
    r6      = _mm256_madd52hi_epu64(r6, inpA[23], inpA[23]); // Add square(47)
    res[47] = r6;
    r7      = _mm256_add_epi64(r7, r7);                      // Double(48)
    r7      = _mm256_madd52lo_epu64(r7, inpA[24], inpA[24]); // Add square(48)
    res[48] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    r4      = zero_simd;
    r5      = zero_simd;
    r6      = zero_simd;
    r7      = zero_simd;
    r8      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[49]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[50]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[51]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[52]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[53]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[54]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[55]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[56]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[49]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[50]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[51]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[52]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[53]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[54]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[55]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[56]);       // Sum(56)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[48]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[49]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[50]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[51]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[52]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[53]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[54]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[55]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[48]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[49]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[50]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[51]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[52]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[53]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[54]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[55]);       // Sum(56)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[47]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[48]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[49]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[50]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[51]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[52]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[53]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[54]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[47]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[48]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[49]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[50]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[51]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[52]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[53]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[54]);       // Sum(56)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[46]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[47]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[48]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[49]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[50]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[51]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[52]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[53]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[46]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[47]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[48]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[49]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[50]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[51]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[52]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[53]);       // Sum(56)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[45]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[46]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[47]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[48]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[49]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[50]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[51]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[52]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[45]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[46]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[47]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[48]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[49]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[50]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[51]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[52]);       // Sum(56)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[44]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[45]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[46]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[47]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[48]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[49]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[50]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[51]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[44]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[45]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[46]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[47]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[48]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[49]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[50]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[51]);       // Sum(56)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[43]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[44]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[45]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[46]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[47]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[48]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[49]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[50]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[43]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[44]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[45]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[46]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[47]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[48]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[49]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[50]);       // Sum(56)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[42]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[43]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[44]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[45]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[46]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[47]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[48]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[49]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[42]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[43]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[44]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[45]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[46]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[47]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[48]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[49]);       // Sum(56)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[43]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[44]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[45]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[46]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[47]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[48]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[43]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[44]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[45]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[46]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[47]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[48]);       // Sum(56)
    AL      = inpA[9];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[40]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[41]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[42]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[43]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[44]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[45]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[46]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[47]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[40]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[41]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[42]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[43]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[44]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[45]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[46]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[47]);       // Sum(56)
    AL      = inpA[10];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[39]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[40]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[41]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[42]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[43]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[44]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[45]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[46]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[39]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[40]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[41]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[42]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[43]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[44]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[45]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[46]);       // Sum(56)
    AL      = inpA[11];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[38]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[39]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[40]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[41]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[42]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[43]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[44]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[45]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[38]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[39]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[40]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[41]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[42]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[43]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[44]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[45]);       // Sum(56)
    AL      = inpA[12];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[39]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[40]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[41]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[42]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[43]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[44]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[39]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[40]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[41]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[42]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[43]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[44]);       // Sum(56)
    AL      = inpA[13];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[39]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[40]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[41]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[42]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[43]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[39]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[40]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[41]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[42]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[43]);       // Sum(56)
    AL      = inpA[14];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[38]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[39]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[40]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[41]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[42]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[38]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[39]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[40]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[41]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[42]);       // Sum(56)
    AL      = inpA[15];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[37]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[38]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[39]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[40]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[41]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[37]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[38]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[39]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[40]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[41]);       // Sum(56)
    AL      = inpA[16];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(56)
    AL      = inpA[17];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[36]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[37]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[38]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[39]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[36]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[37]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[38]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[39]);       // Sum(56)
    AL      = inpA[18];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[37]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[38]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[37]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[38]);       // Sum(56)
    AL      = inpA[19];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(56)
    AL      = inpA[20];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(56)
    AL      = inpA[21];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(56)
    AL      = inpA[22];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(56)
    AL      = inpA[23];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(56)
    AL      = inpA[24];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(49)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(50)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(56)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(49)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(50)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(56)
    AL      = inpA[25];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(51)
    r3      = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(52)
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(56)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(51)
    r4      = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(52)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(56)
    AL      = inpA[26];
    r4      = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(53)
    r5      = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(54)
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(56)
    r5      = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(53)
    r6      = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(54)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(56)
    AL      = inpA[27];
    r6      = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(55)
    r7      = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(56)
    r7      = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(55)
    r8      = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(56)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(49)
    r0      = _mm256_madd52hi_epu64(r0, inpA[24], inpA[24]); // Add square(49)
    res[49] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(50)
    r1      = _mm256_madd52lo_epu64(r1, inpA[25], inpA[25]); // Add square(50)
    res[50] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(51)
    r2      = _mm256_madd52hi_epu64(r2, inpA[25], inpA[25]); // Add square(51)
    res[51] = r2;
    r3      = _mm256_add_epi64(r3, r3);                      // Double(52)
    r3      = _mm256_madd52lo_epu64(r3, inpA[26], inpA[26]); // Add square(52)
    res[52] = r3;
    r4      = _mm256_add_epi64(r4, r4);                      // Double(53)
    r4      = _mm256_madd52hi_epu64(r4, inpA[26], inpA[26]); // Add square(53)
    res[53] = r4;
    r5      = _mm256_add_epi64(r5, r5);                      // Double(54)
    r5      = _mm256_madd52lo_epu64(r5, inpA[27], inpA[27]); // Add square(54)
    res[54] = r5;
    r6      = _mm256_add_epi64(r6, r6);                      // Double(55)
    r6      = _mm256_madd52hi_epu64(r6, inpA[27], inpA[27]); // Add square(55)
    res[55] = r6;
    r7      = _mm256_add_epi64(r7, r7);                      // Double(56)
    r7      = _mm256_madd52lo_epu64(r7, inpA[28], inpA[28]); // Add square(56)
    res[56] = r7;
    r0      = r8;
    r1      = zero_simd;
    r2      = zero_simd;
    r3      = zero_simd;
    AL      = inpA[0];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[57]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[58]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[59]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[57]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[58]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[59]);       // Sum(59)
    AL      = inpA[1];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[56]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[57]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[58]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[56]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[57]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[58]);       // Sum(59)
    AL      = inpA[2];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[55]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[56]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[57]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[55]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[56]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[57]);       // Sum(59)
    AL      = inpA[3];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[54]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[55]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[56]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[54]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[55]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[56]);       // Sum(59)
    AL      = inpA[4];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[53]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[54]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[55]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[53]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[54]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[55]);       // Sum(59)
    AL      = inpA[5];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[52]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[53]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[54]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[52]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[53]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[54]);       // Sum(59)
    AL      = inpA[6];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[51]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[52]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[53]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[51]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[52]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[53]);       // Sum(59)
    AL      = inpA[7];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[50]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[51]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[52]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[50]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[51]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[52]);       // Sum(59)
    AL      = inpA[8];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[49]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[50]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[51]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[49]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[50]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[51]);       // Sum(59)
    AL      = inpA[9];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[48]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[49]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[50]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[48]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[49]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[50]);       // Sum(59)
    AL      = inpA[10];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[47]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[48]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[49]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[47]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[48]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[49]);       // Sum(59)
    AL      = inpA[11];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[46]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[47]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[48]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[46]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[47]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[48]);       // Sum(59)
    AL      = inpA[12];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[45]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[46]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[47]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[45]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[46]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[47]);       // Sum(59)
    AL      = inpA[13];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[44]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[45]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[46]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[44]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[45]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[46]);       // Sum(59)
    AL      = inpA[14];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[43]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[44]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[45]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[43]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[44]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[45]);       // Sum(59)
    AL      = inpA[15];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[42]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[43]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[44]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[42]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[43]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[44]);       // Sum(59)
    AL      = inpA[16];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[43]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[43]);       // Sum(59)
    AL      = inpA[17];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[40]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[41]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[42]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[40]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[41]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[42]);       // Sum(59)
    AL      = inpA[18];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[39]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[40]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[41]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[39]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[40]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[41]);       // Sum(59)
    AL      = inpA[19];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[38]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[39]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[40]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[38]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[39]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[40]);       // Sum(59)
    AL      = inpA[20];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[39]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[39]);       // Sum(59)
    AL      = inpA[21];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(59)
    AL      = inpA[22];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(59)
    AL      = inpA[23];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(59)
    AL      = inpA[24];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(59)
    AL      = inpA[25];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(59)
    AL      = inpA[26];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(59)
    AL      = inpA[27];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(59)
    AL      = inpA[28];
    r0      = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(57)
    r1      = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(58)
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(59)
    r1      = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(57)
    r2      = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(58)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(59)
    AL      = inpA[29];
    r2      = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(59)
    r3      = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(59)
    r0      = _mm256_add_epi64(r0, r0);                      // Double(57)
    r0      = _mm256_madd52hi_epu64(r0, inpA[28], inpA[28]); // Add square(57)
    res[57] = r0;
    r1      = _mm256_add_epi64(r1, r1);                      // Double(58)
    r1      = _mm256_madd52lo_epu64(r1, inpA[29], inpA[29]); // Add square(58)
    res[58] = r1;
    r2      = _mm256_add_epi64(r2, r2);                      // Double(59)
    r2      = _mm256_madd52hi_epu64(r2, inpA[29], inpA[29]); // Add square(59)
    res[59] = r2;
    r0      = r3;                                            // finish up 1st triangle

    ASM("jmp l0\nl0:\n");

    // 2nd triangle - sum the products, double and square
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[1]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[2]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[3]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[4]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[5]);        // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[6]);        // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[7]);        // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[8]);        // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[1]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[2]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[3]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[4]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[5]);        // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[6]);        // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[7]);        // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[8]);        // Sum(68)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[2]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[3]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[4]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[5]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[6]);        // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[7]);        // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[8]);        // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[9]);        // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[2]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[3]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[4]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[5]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[6]);        // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[7]);        // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[8]);        // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[9]);        // Sum(68)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[3]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[4]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[5]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[6]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[7]);        // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[8]);        // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[9]);        // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[10]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[3]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[4]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[5]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[6]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[7]);        // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[8]);        // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[9]);        // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[10]);       // Sum(68)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[4]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[5]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[6]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[7]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[8]);        // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[9]);        // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[10]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[11]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[4]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[5]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[6]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[7]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[8]);        // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[9]);        // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[10]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[11]);       // Sum(68)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[5]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[6]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[7]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[8]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[9]);        // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[10]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[11]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[12]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[5]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[6]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[7]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[8]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[9]);        // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[10]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[11]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[12]);       // Sum(68)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[6]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[7]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[8]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[9]);        // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[10]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[11]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[12]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[13]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[6]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[7]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[8]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[9]);        // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[10]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[11]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[12]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[13]);       // Sum(68)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[7]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[8]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[9]);        // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[10]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[11]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[12]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[13]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[14]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[7]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[8]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[9]);        // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[10]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[11]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[12]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[13]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[14]);       // Sum(68)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[8]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[9]);        // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[10]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[11]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[12]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[13]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[14]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[15]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[8]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[9]);        // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[10]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[11]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[12]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[13]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[14]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[15]);       // Sum(68)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[9]);        // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[10]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[11]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[12]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[13]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[14]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[15]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[16]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[9]);        // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[10]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[11]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[12]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[13]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[14]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[15]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[16]);       // Sum(68)
    AL          = inpA[50];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[10]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[11]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[12]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[13]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[14]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[15]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[16]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[17]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[10]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[11]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[12]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[13]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[14]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[15]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[16]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[17]);       // Sum(68)
    AL          = inpA[49];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[11]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[12]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[13]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[14]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[15]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[16]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[17]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[18]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[11]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[12]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[13]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[14]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[15]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[16]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[17]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[18]);       // Sum(68)
    AL          = inpA[48];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[12]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[13]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[14]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[15]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[16]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[17]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[18]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[19]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[12]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[13]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[14]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[15]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[16]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[17]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[18]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[19]);       // Sum(68)
    AL          = inpA[47];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[13]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[14]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[15]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[16]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[17]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[18]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[19]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[20]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[13]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[14]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[15]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[16]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[17]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[18]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[19]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[20]);       // Sum(68)
    AL          = inpA[46];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[14]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[15]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[16]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[17]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[18]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[19]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[20]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[21]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[14]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[15]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[16]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[17]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[18]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[19]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[20]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[21]);       // Sum(68)
    AL          = inpA[45];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[15]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[16]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[17]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[18]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[19]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[20]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[21]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[22]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[15]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[16]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[17]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[18]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[19]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[20]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[21]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[22]);       // Sum(68)
    AL          = inpA[44];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[16]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[17]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[18]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[19]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[20]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[21]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[22]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[23]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[16]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[17]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[18]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[19]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[20]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[21]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[22]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[23]);       // Sum(68)
    AL          = inpA[43];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(68)
    AL          = inpA[42];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[18]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[19]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[20]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[21]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[22]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[23]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[18]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[19]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[20]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[21]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[22]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[23]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(68)
    AL          = inpA[41];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[19]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[20]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[21]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[22]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[19]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[20]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[21]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[22]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(68)
    AL          = inpA[40];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[20]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[21]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[20]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[21]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(68)
    AL          = inpA[39];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(68)
    AL          = inpA[38];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(68)
    AL          = inpA[37];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(68)
    AL          = inpA[36];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(68)
    AL          = inpA[35];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(68)
    AL          = inpA[34];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(66)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(67)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(68)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(66)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(67)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(68)
    AL          = inpA[33];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(64)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(65)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(66)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(64)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(65)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(66)
    AL          = inpA[32];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(62)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(63)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(64)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(62)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(63)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(64)
    AL          = inpA[31];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(61)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(62)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(61)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(62)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(60)
    r0          = _mm256_madd52lo_epu64(r0, inpA[30], inpA[30]); // Add square(60)
    res[N + 0]  = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(61)
    r1          = _mm256_madd52hi_epu64(r1, inpA[30], inpA[30]); // Add square(61)
    res[N + 1]  = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(62)
    r2          = _mm256_madd52lo_epu64(r2, inpA[31], inpA[31]); // Add square(62)
    res[N + 2]  = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(63)
    r3          = _mm256_madd52hi_epu64(r3, inpA[31], inpA[31]); // Add square(63)
    res[N + 3]  = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(64)
    r4          = _mm256_madd52lo_epu64(r4, inpA[32], inpA[32]); // Add square(64)
    res[N + 4]  = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(65)
    r5          = _mm256_madd52hi_epu64(r5, inpA[32], inpA[32]); // Add square(65)
    res[N + 5]  = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(66)
    r6          = _mm256_madd52lo_epu64(r6, inpA[33], inpA[33]); // Add square(66)
    res[N + 6]  = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(67)
    r7          = _mm256_madd52hi_epu64(r7, inpA[33], inpA[33]); // Add square(67)
    res[N + 7]  = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[9]);        // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[10]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[11]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[12]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[13]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[14]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[15]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[16]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[9]);        // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[10]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[11]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[12]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[13]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[14]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[15]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[16]);       // Sum(76)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[10]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[11]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[12]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[13]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[14]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[15]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[16]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[17]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[10]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[11]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[12]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[13]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[14]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[15]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[16]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[17]);       // Sum(76)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[11]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[12]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[13]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[14]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[15]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[16]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[17]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[18]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[11]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[12]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[13]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[14]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[15]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[16]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[17]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[18]);       // Sum(76)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[12]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[13]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[14]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[15]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[16]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[17]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[18]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[19]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[12]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[13]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[14]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[15]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[16]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[17]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[18]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[19]);       // Sum(76)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[13]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[14]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[15]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[16]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[17]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[18]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[19]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[20]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[13]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[14]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[15]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[16]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[17]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[18]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[19]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[20]);       // Sum(76)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[14]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[15]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[16]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[17]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[18]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[19]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[20]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[21]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[14]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[15]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[16]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[17]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[18]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[19]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[20]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[21]);       // Sum(76)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[15]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[16]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[17]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[18]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[19]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[20]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[21]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[22]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[15]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[16]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[17]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[18]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[19]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[20]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[21]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[22]);       // Sum(76)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[16]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[17]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[18]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[19]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[20]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[21]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[22]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[23]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[16]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[17]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[18]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[19]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[20]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[21]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[22]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[23]);       // Sum(76)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(76)
    AL          = inpA[50];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[18]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[19]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[20]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[21]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[22]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[23]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[18]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[19]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[20]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[21]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[22]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[23]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(76)
    AL          = inpA[49];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[19]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[20]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[21]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[22]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[19]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[20]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[21]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[22]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(76)
    AL          = inpA[48];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[20]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[21]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[20]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[21]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(76)
    AL          = inpA[47];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(76)
    AL          = inpA[46];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(76)
    AL          = inpA[45];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(76)
    AL          = inpA[44];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(76)
    AL          = inpA[43];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(76)
    AL          = inpA[42];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(76)
    AL          = inpA[41];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(76)
    AL          = inpA[40];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(76)
    AL          = inpA[39];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(76)
    AL          = inpA[38];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(74)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(75)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(76)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(74)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(75)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(76)
    AL          = inpA[37];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(72)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(73)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(74)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(72)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(73)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(74)
    AL          = inpA[36];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(70)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(71)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(72)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(70)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(71)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(72)
    AL          = inpA[35];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(69)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(70)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(69)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(70)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(68)
    r0          = _mm256_madd52lo_epu64(r0, inpA[34], inpA[34]); // Add square(68)
    res[N + 8]  = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(69)
    r1          = _mm256_madd52hi_epu64(r1, inpA[34], inpA[34]); // Add square(69)
    res[N + 9]  = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(70)
    r2          = _mm256_madd52lo_epu64(r2, inpA[35], inpA[35]); // Add square(70)
    res[N + 10] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(71)
    r3          = _mm256_madd52hi_epu64(r3, inpA[35], inpA[35]); // Add square(71)
    res[N + 11] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(72)
    r4          = _mm256_madd52lo_epu64(r4, inpA[36], inpA[36]); // Add square(72)
    res[N + 12] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(73)
    r5          = _mm256_madd52hi_epu64(r5, inpA[36], inpA[36]); // Add square(73)
    res[N + 13] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(74)
    r6          = _mm256_madd52lo_epu64(r6, inpA[37], inpA[37]); // Add square(74)
    res[N + 14] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(75)
    r7          = _mm256_madd52hi_epu64(r7, inpA[37], inpA[37]); // Add square(75)
    res[N + 15] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[17]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[18]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[19]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[20]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[21]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[22]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[23]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[24]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[17]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[18]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[19]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[20]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[21]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[22]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[23]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[24]);       // Sum(84)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[18]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[19]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[20]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[21]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[22]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[23]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[24]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[25]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[18]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[19]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[20]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[21]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[22]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[23]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[24]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[25]);       // Sum(84)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[19]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[20]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[21]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[22]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[23]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[24]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[25]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[26]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[19]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[20]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[21]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[22]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[23]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[24]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[25]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[26]);       // Sum(84)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[20]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[21]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[22]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[23]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[24]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[25]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[26]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[27]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[20]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[21]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[22]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[23]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[24]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[25]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[26]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[27]);       // Sum(84)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[21]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[22]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[23]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[24]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[25]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[26]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[27]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[28]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[21]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[22]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[23]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[24]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[25]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[26]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[27]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[28]);       // Sum(84)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[22]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[23]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[24]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[25]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[26]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[27]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[28]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[29]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[22]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[23]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[24]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[25]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[26]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[27]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[28]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[29]);       // Sum(84)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[23]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[24]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[25]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[26]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[27]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[28]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[29]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[30]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[23]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[24]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[25]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[26]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[27]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[28]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[29]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[30]);       // Sum(84)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[24]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[25]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[26]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[27]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[28]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[29]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[30]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[31]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[24]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[25]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[26]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[27]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[28]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[29]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[30]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[31]);       // Sum(84)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(84)
    AL          = inpA[50];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(84)
    AL          = inpA[49];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(84)
    AL          = inpA[48];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(84)
    AL          = inpA[47];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(84)
    AL          = inpA[46];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(84)
    AL          = inpA[45];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[37]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[38]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[37]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[38]);       // Sum(84)
    AL          = inpA[44];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[36]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[37]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[38]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[39]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[36]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[37]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[38]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[39]);       // Sum(84)
    AL          = inpA[43];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(84)
    AL          = inpA[42];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[37]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[38]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[39]);       // Sum(82)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[40]);       // Sum(83)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[41]);       // Sum(84)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[37]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[38]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[39]);       // Sum(82)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[40]);       // Sum(83)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[41]);       // Sum(84)
    AL          = inpA[41];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[38]);       // Sum(80)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[39]);       // Sum(81)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[40]);       // Sum(82)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[38]);       // Sum(80)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[39]);       // Sum(81)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[40]);       // Sum(82)
    AL          = inpA[40];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(78)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(79)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[39]);       // Sum(80)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(78)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(79)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[39]);       // Sum(80)
    AL          = inpA[39];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(77)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(78)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(77)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(78)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(76)
    r0          = _mm256_madd52lo_epu64(r0, inpA[38], inpA[38]); // Add square(76)
    res[N + 16] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(77)
    r1          = _mm256_madd52hi_epu64(r1, inpA[38], inpA[38]); // Add square(77)
    res[N + 17] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(78)
    r2          = _mm256_madd52lo_epu64(r2, inpA[39], inpA[39]); // Add square(78)
    res[N + 18] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(79)
    r3          = _mm256_madd52hi_epu64(r3, inpA[39], inpA[39]); // Add square(79)
    res[N + 19] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(80)
    r4          = _mm256_madd52lo_epu64(r4, inpA[40], inpA[40]); // Add square(80)
    res[N + 20] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(81)
    r5          = _mm256_madd52hi_epu64(r5, inpA[40], inpA[40]); // Add square(81)
    res[N + 21] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(82)
    r6          = _mm256_madd52lo_epu64(r6, inpA[41], inpA[41]); // Add square(82)
    res[N + 22] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(83)
    r7          = _mm256_madd52hi_epu64(r7, inpA[41], inpA[41]); // Add square(83)
    res[N + 23] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[25]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[26]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[27]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[28]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[29]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[30]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[31]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[32]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[25]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[26]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[27]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[28]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[29]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[30]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[31]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[32]);       // Sum(92)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[26]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[27]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[28]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[29]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[30]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[31]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[32]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[33]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[26]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[27]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[28]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[29]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[30]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[31]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[32]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[33]);       // Sum(92)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[27]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[28]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[29]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[30]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[31]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[32]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[33]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[34]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[27]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[28]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[29]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[30]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[31]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[32]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[33]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[34]);       // Sum(92)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[28]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[29]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[30]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[31]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[32]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[33]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[34]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[35]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[28]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[29]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[30]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[31]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[32]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[33]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[34]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[35]);       // Sum(92)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[29]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[30]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[31]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[32]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[33]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[34]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[35]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[36]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[29]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[30]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[31]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[32]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[33]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[34]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[35]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[36]);       // Sum(92)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[30]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[31]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[32]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[33]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[34]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[35]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[36]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[37]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[30]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[31]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[32]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[33]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[34]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[35]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[36]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[37]);       // Sum(92)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[31]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[32]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[33]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[34]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[35]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[36]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[37]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[38]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[31]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[32]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[33]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[34]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[35]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[36]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[37]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[38]);       // Sum(92)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[32]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[33]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[34]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[35]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[36]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[37]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[38]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[39]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[32]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[33]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[34]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[35]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[36]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[37]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[38]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[39]);       // Sum(92)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(92)
    AL          = inpA[50];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[37]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[38]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[39]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[40]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[41]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[37]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[38]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[39]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[40]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[41]);       // Sum(92)
    AL          = inpA[49];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[38]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[39]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[40]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[41]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[42]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[38]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[39]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[40]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[41]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[42]);       // Sum(92)
    AL          = inpA[48];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[39]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[40]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[41]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[42]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[43]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[39]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[40]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[41]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[42]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[43]);       // Sum(92)
    AL          = inpA[47];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[39]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[40]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[41]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[42]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[43]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[44]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[39]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[40]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[41]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[42]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[43]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[44]);       // Sum(92)
    AL          = inpA[46];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[38]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[39]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[40]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[41]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[42]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[43]);       // Sum(90)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[44]);       // Sum(91)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[45]);       // Sum(92)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[38]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[39]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[40]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[41]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[42]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[43]);       // Sum(90)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[44]);       // Sum(91)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[45]);       // Sum(92)
    AL          = inpA[45];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[39]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[40]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[41]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[42]);       // Sum(88)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[43]);       // Sum(89)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[44]);       // Sum(90)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[39]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[40]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[41]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[42]);       // Sum(88)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[43]);       // Sum(89)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[44]);       // Sum(90)
    AL          = inpA[44];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[40]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[41]);       // Sum(86)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[42]);       // Sum(87)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[43]);       // Sum(88)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[40]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[41]);       // Sum(86)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[42]);       // Sum(87)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[43]);       // Sum(88)
    AL          = inpA[43];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(85)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(86)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(85)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(86)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(84)
    r0          = _mm256_madd52lo_epu64(r0, inpA[42], inpA[42]); // Add square(84)
    res[N + 24] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(85)
    r1          = _mm256_madd52hi_epu64(r1, inpA[42], inpA[42]); // Add square(85)
    res[N + 25] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(86)
    r2          = _mm256_madd52lo_epu64(r2, inpA[43], inpA[43]); // Add square(86)
    res[N + 26] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(87)
    r3          = _mm256_madd52hi_epu64(r3, inpA[43], inpA[43]); // Add square(87)
    res[N + 27] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(88)
    r4          = _mm256_madd52lo_epu64(r4, inpA[44], inpA[44]); // Add square(88)
    res[N + 28] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(89)
    r5          = _mm256_madd52hi_epu64(r5, inpA[44], inpA[44]); // Add square(89)
    res[N + 29] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(90)
    r6          = _mm256_madd52lo_epu64(r6, inpA[45], inpA[45]); // Add square(90)
    res[N + 30] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(91)
    r7          = _mm256_madd52hi_epu64(r7, inpA[45], inpA[45]); // Add square(91)
    res[N + 31] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[33]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[34]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[35]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[36]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[37]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[38]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[39]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[40]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[33]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[34]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[35]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[36]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[37]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[38]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[39]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[40]);       // Sum(100)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[34]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[35]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[36]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[37]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[38]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[39]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[40]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[41]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[34]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[35]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[36]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[37]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[38]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[39]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[40]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[41]);       // Sum(100)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[35]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[36]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[37]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[38]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[39]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[40]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[41]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[42]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[35]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[36]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[37]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[38]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[39]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[40]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[41]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[42]);       // Sum(100)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[36]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[37]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[38]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[39]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[40]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[41]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[42]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[43]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[36]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[37]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[38]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[39]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[40]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[41]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[42]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[43]);       // Sum(100)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[37]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[38]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[39]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[40]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[41]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[42]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[43]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[44]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[37]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[38]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[39]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[40]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[41]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[42]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[43]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[44]);       // Sum(100)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[38]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[39]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[40]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[41]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[42]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[43]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[44]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[45]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[38]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[39]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[40]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[41]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[42]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[43]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[44]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[45]);       // Sum(100)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[39]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[40]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[41]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[42]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[43]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[44]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[45]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[46]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[39]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[40]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[41]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[42]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[43]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[44]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[45]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[46]);       // Sum(100)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[40]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[41]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[42]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[43]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[44]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[45]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[46]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[47]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[40]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[41]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[42]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[43]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[44]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[45]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[46]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[47]);       // Sum(100)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[43]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[44]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[45]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[46]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[47]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[48]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[43]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[44]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[45]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[46]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[47]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[48]);       // Sum(100)
    AL          = inpA[50];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[42]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[43]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[44]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[45]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[46]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[47]);       // Sum(98)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[48]);       // Sum(99)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[49]);       // Sum(100)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[42]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[43]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[44]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[45]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[46]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[47]);       // Sum(98)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[48]);       // Sum(99)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[49]);       // Sum(100)
    AL          = inpA[49];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[43]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[44]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[45]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[46]);       // Sum(96)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[47]);       // Sum(97)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[48]);       // Sum(98)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[43]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[44]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[45]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[46]);       // Sum(96)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[47]);       // Sum(97)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[48]);       // Sum(98)
    AL          = inpA[48];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[44]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[45]);       // Sum(94)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[46]);       // Sum(95)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[47]);       // Sum(96)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[44]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[45]);       // Sum(94)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[46]);       // Sum(95)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[47]);       // Sum(96)
    AL          = inpA[47];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[45]);       // Sum(93)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[46]);       // Sum(94)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[45]);       // Sum(93)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[46]);       // Sum(94)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(92)
    r0          = _mm256_madd52lo_epu64(r0, inpA[46], inpA[46]); // Add square(92)
    res[N + 32] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(93)
    r1          = _mm256_madd52hi_epu64(r1, inpA[46], inpA[46]); // Add square(93)
    res[N + 33] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(94)
    r2          = _mm256_madd52lo_epu64(r2, inpA[47], inpA[47]); // Add square(94)
    res[N + 34] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(95)
    r3          = _mm256_madd52hi_epu64(r3, inpA[47], inpA[47]); // Add square(95)
    res[N + 35] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(96)
    r4          = _mm256_madd52lo_epu64(r4, inpA[48], inpA[48]); // Add square(96)
    res[N + 36] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(97)
    r5          = _mm256_madd52hi_epu64(r5, inpA[48], inpA[48]); // Add square(97)
    res[N + 37] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(98)
    r6          = _mm256_madd52lo_epu64(r6, inpA[49], inpA[49]); // Add square(98)
    res[N + 38] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(99)
    r7          = _mm256_madd52hi_epu64(r7, inpA[49], inpA[49]); // Add square(99)
    res[N + 39] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[41]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[42]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[43]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[44]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[45]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[46]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[47]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[48]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[41]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[42]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[43]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[44]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[45]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[46]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[47]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[48]);       // Sum(108)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[42]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[43]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[44]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[45]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[46]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[47]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[48]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[49]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[42]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[43]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[44]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[45]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[46]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[47]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[48]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[49]);       // Sum(108)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[43]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[44]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[45]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[46]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[47]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[48]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[49]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[50]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[43]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[44]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[45]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[46]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[47]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[48]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[49]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[50]);       // Sum(108)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[44]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[45]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[46]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[47]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[48]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[49]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[50]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[51]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[44]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[45]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[46]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[47]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[48]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[49]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[50]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[51]);       // Sum(108)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[45]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[46]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[47]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[48]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[49]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[50]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[51]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[52]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[45]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[46]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[47]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[48]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[49]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[50]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[51]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[52]);       // Sum(108)
    AL          = inpA[54];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[46]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[47]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[48]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[49]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[50]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[51]);       // Sum(106)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[52]);       // Sum(107)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[53]);       // Sum(108)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[46]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[47]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[48]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[49]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[50]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[51]);       // Sum(106)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[52]);       // Sum(107)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[53]);       // Sum(108)
    AL          = inpA[53];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[47]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[48]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[49]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[50]);       // Sum(104)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[51]);       // Sum(105)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[52]);       // Sum(106)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[47]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[48]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[49]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[50]);       // Sum(104)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[51]);       // Sum(105)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[52]);       // Sum(106)
    AL          = inpA[52];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[48]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[49]);       // Sum(102)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[50]);       // Sum(103)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[51]);       // Sum(104)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[48]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[49]);       // Sum(102)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[50]);       // Sum(103)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[51]);       // Sum(104)
    AL          = inpA[51];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[49]);       // Sum(101)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[50]);       // Sum(102)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[49]);       // Sum(101)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[50]);       // Sum(102)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(100)
    r0          = _mm256_madd52lo_epu64(r0, inpA[50], inpA[50]); // Add square(100)
    res[N + 40] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(101)
    r1          = _mm256_madd52hi_epu64(r1, inpA[50], inpA[50]); // Add square(101)
    res[N + 41] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(102)
    r2          = _mm256_madd52lo_epu64(r2, inpA[51], inpA[51]); // Add square(102)
    res[N + 42] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(103)
    r3          = _mm256_madd52hi_epu64(r3, inpA[51], inpA[51]); // Add square(103)
    res[N + 43] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(104)
    r4          = _mm256_madd52lo_epu64(r4, inpA[52], inpA[52]); // Add square(104)
    res[N + 44] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(105)
    r5          = _mm256_madd52hi_epu64(r5, inpA[52], inpA[52]); // Add square(105)
    res[N + 45] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(106)
    r6          = _mm256_madd52lo_epu64(r6, inpA[53], inpA[53]); // Add square(106)
    res[N + 46] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(107)
    r7          = _mm256_madd52hi_epu64(r7, inpA[53], inpA[53]); // Add square(107)
    res[N + 47] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    r4          = zero_simd;
    r5          = zero_simd;
    r6          = zero_simd;
    r7          = zero_simd;
    r8          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[49]);       // Sum(109)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[50]);       // Sum(110)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[51]);       // Sum(111)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[52]);       // Sum(112)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[53]);       // Sum(113)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[54]);       // Sum(114)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[55]);       // Sum(115)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[56]);       // Sum(116)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[49]);       // Sum(109)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[50]);       // Sum(110)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[51]);       // Sum(111)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[52]);       // Sum(112)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[53]);       // Sum(113)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[54]);       // Sum(114)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[55]);       // Sum(115)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[56]);       // Sum(116)
    AL          = inpA[58];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[50]);       // Sum(109)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[51]);       // Sum(110)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[52]);       // Sum(111)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[53]);       // Sum(112)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[54]);       // Sum(113)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[55]);       // Sum(114)
    r6          = _mm256_madd52lo_epu64(r6, AL, inpA[56]);       // Sum(115)
    r7          = _mm256_madd52lo_epu64(r7, AL, inpA[57]);       // Sum(116)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[50]);       // Sum(109)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[51]);       // Sum(110)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[52]);       // Sum(111)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[53]);       // Sum(112)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[54]);       // Sum(113)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[55]);       // Sum(114)
    r7          = _mm256_madd52hi_epu64(r7, AL, inpA[56]);       // Sum(115)
    r8          = _mm256_madd52hi_epu64(r8, AL, inpA[57]);       // Sum(116)
    AL          = inpA[57];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[51]);       // Sum(109)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[52]);       // Sum(110)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[53]);       // Sum(111)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[54]);       // Sum(112)
    r4          = _mm256_madd52lo_epu64(r4, AL, inpA[55]);       // Sum(113)
    r5          = _mm256_madd52lo_epu64(r5, AL, inpA[56]);       // Sum(114)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[51]);       // Sum(109)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[52]);       // Sum(110)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[53]);       // Sum(111)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[54]);       // Sum(112)
    r5          = _mm256_madd52hi_epu64(r5, AL, inpA[55]);       // Sum(113)
    r6          = _mm256_madd52hi_epu64(r6, AL, inpA[56]);       // Sum(114)
    AL          = inpA[56];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[52]);       // Sum(109)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[53]);       // Sum(110)
    r2          = _mm256_madd52lo_epu64(r2, AL, inpA[54]);       // Sum(111)
    r3          = _mm256_madd52lo_epu64(r3, AL, inpA[55]);       // Sum(112)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[52]);       // Sum(109)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[53]);       // Sum(110)
    r3          = _mm256_madd52hi_epu64(r3, AL, inpA[54]);       // Sum(111)
    r4          = _mm256_madd52hi_epu64(r4, AL, inpA[55]);       // Sum(112)
    AL          = inpA[55];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[53]);       // Sum(109)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[54]);       // Sum(110)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[53]);       // Sum(109)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[54]);       // Sum(110)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(108)
    r0          = _mm256_madd52lo_epu64(r0, inpA[54], inpA[54]); // Add square(108)
    res[N + 48] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(109)
    r1          = _mm256_madd52hi_epu64(r1, inpA[54], inpA[54]); // Add square(109)
    res[N + 49] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(110)
    r2          = _mm256_madd52lo_epu64(r2, inpA[55], inpA[55]); // Add square(110)
    res[N + 50] = r2;
    r3          = _mm256_add_epi64(r3, r3);                      // Double(111)
    r3          = _mm256_madd52hi_epu64(r3, inpA[55], inpA[55]); // Add square(111)
    res[N + 51] = r3;
    r4          = _mm256_add_epi64(r4, r4);                      // Double(112)
    r4          = _mm256_madd52lo_epu64(r4, inpA[56], inpA[56]); // Add square(112)
    res[N + 52] = r4;
    r5          = _mm256_add_epi64(r5, r5);                      // Double(113)
    r5          = _mm256_madd52hi_epu64(r5, inpA[56], inpA[56]); // Add square(113)
    res[N + 53] = r5;
    r6          = _mm256_add_epi64(r6, r6);                      // Double(114)
    r6          = _mm256_madd52lo_epu64(r6, inpA[57], inpA[57]); // Add square(114)
    res[N + 54] = r6;
    r7          = _mm256_add_epi64(r7, r7);                      // Double(115)
    r7          = _mm256_madd52hi_epu64(r7, inpA[57], inpA[57]); // Add square(115)
    res[N + 55] = r7;
    r0          = r8;
    r1          = zero_simd;
    r2          = zero_simd;
    r3          = zero_simd;
    AL          = inpA[59];
    r0          = _mm256_madd52lo_epu64(r0, AL, inpA[57]);       // Sum(117)
    r1          = _mm256_madd52lo_epu64(r1, AL, inpA[58]);       // Sum(118)
    r1          = _mm256_madd52hi_epu64(r1, AL, inpA[57]);       // Sum(117)
    r2          = _mm256_madd52hi_epu64(r2, AL, inpA[58]);       // Sum(118)
    r0          = _mm256_add_epi64(r0, r0);                      // Double(116)
    r0          = _mm256_madd52lo_epu64(r0, inpA[58], inpA[58]); // Add square(116)
    res[N + 56] = r0;
    r1          = _mm256_add_epi64(r1, r1);                      // Double(117)
    r1          = _mm256_madd52hi_epu64(r1, inpA[58], inpA[58]); // Add square(117)
    res[N + 57] = r1;
    r2          = _mm256_add_epi64(r2, r2);                      // Double(118)
    r2          = _mm256_madd52lo_epu64(r2, inpA[59], inpA[59]); // Add square(118)
    res[N + 58] = r2;
    r0          = r3;
    // finish up doubling
    res[N + 59] = _mm256_madd52hi_epu64(zero_simd, inpA[59], inpA[59]);
}

void AMS52x60_diagonal_mb4(int64u* out_mb,
                           const int64u* inpA_mb,
                           const int64u* inpM_mb,
                           const int64u* k0_mb)
{
    const int N = 60;
    __m256i res[60 * 2];

    /* Square only */
    ams52x60_square_diagonal_mb4(res, inpA_mb);

    /* Generate u_i and begin reduction */
    ams_reduce_52xN_mb4(res, inpM_mb, k0_mb, N);

    /* Normalize */
    ifma_normalize_ams_52xN_mb4(out_mb, res, N);
}

#endif // #if ((_MBX == _MBX_L9) && _MBX_AVX_IFMA_SUPPORTED)
