// bibiman - a TUI for managing BibLaTeX databases
// Copyright (C) 2025  lukeflo
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <https://www.gnu.org/licenses/>.
/////

use biblatex::{ChunksExt, Entry, Type};
use indoc::formatdoc;
use owo_colors::{
    OwoColorize,
    colors::{BrightBlue, Green, White},
};

use crate::bibiman::{citekeys::CitekeyCase, sanitize::sanitize_single_string_fully};

pub(super) const SKIPPED_ENTRIES: [&str; 2] = ["set", "xdata"];

pub(super) fn formatting_help() {
    let help = vec![
        formatdoc!(
            "{} {}\n",
            env!("CARGO_PKG_NAME").fg::<Green>().bold(),
            env!("CARGO_PKG_VERSION")
        ),
        formatdoc!("{}", "USAGE".bold()),
        formatdoc!(
            "\t{} {} {} {} {}\n",
            env!("CARGO_PKG_NAME").fg::<White>().bold(),
            "format-citekeys".bold(),
            "[OPTIONS]".bold(),
            "--source=<SOURCE>".bold(),
            "[--output=<TARGET>]".bold()
        ),
        formatdoc!(
            "
                \tThis help describes the CLI usage for the citekey formatting
                \tfunctionality of bibiman. The definition of patterns how the
                \tcitekeys should be formatted must be set in the config file.
                \tFor further informations how to use this patterns etc. see:
                \t{}
            ",
            "https://codeberg.org/lukeflo/bibiman/src/branch/main/CITEKEYS.md"
                .italic()
                .fg::<BrightBlue>()
        ),
        formatdoc!("{}", "OPTIONS".bold()),
        formatdoc!(
            "
                \t{}
                \tShow this help and exit
            ",
            "-h, --help".fg::<White>().bold()
        ),
        formatdoc!(
            "
                \t{}
                \tDon't apply any changes to the named files. Instead print all
                \told citekeys and the formatted strings that would have been
                \tapplied in the format: {} => {}
            ",
            "-d, --dry-run".fg::<White>().bold(),
            "old_key".italic(),
            "new_key".bold()
        ),
        formatdoc! {"
                \t{}
                \tThe bibfile for which the citekey formatting should be processed.
                \tTakes a path as argument.
            ", "-s, -f, --source=<PATH>, --file=<PATH>".fg::<White>().bold()},
        formatdoc!(
            "
                \t{}
                \tThe bibfile to which the updated content should be written.
                \tTakes a path as argument. If the file doesn't exist, it will be
                \tcreated.
                \tIf the argument isn't used, the original file will be {}!
            ",
            "-t, -o, --target=<PATH>, --output=<PATH>"
                .fg::<White>()
                .bold(),
            "overwritten".italic(),
        ),
        formatdoc!(
            "
                \t{}
                \tWhen this option is set, bibiman will also rename all PDFs and 
                \tnotefiles following the bibiman citekey-basename scheme at the 
                \tlocations set in the config file. This option can break file paths. 
                \tTry with {} first!
            ",
            "-u, --update-attachments".fg::<White>().bold(),
            "--dry-run".bold()
        ),
    ];
    let help = help.join("\n");
    println!("{}", help);
}

/// Build the citekey from the patterns defined in the config file
pub(super) fn build_citekey(
    entry: &Entry,
    pattern_fields: &[String],
    case: Option<&CitekeyCase>,
    ascii_only: bool,
    ignored_chars: &[char],
    ignored_words: &[String],
) -> String {
    // mut string the citekey is built from
    let mut new_citekey = String::new();

    // trailing delimiter of previous field
    let mut trailing_delimiter: Option<&str> = None;

    // loop over pattern fields process them
    'field_loop: for pattern in pattern_fields.iter() {
        // parse single values from pattern field
        let (field_name, max_words, max_chars, inner_delimiter, cur_trailing_delimiter) =
            split_formatting_pat(pattern);

        // built the part of the citekey from the current pattern field
        let formatted_field_str = {
            let mut formatted_str = String::new();

            // preformat the field depending on biblatex value
            let field = preformat_field(field_name, entry);

            // split at whitespaces, count fields and set counter for processed
            // splits
            let split_field = field.split_whitespace();
            let mut words_passed = 0;
            let field_count = field.split_whitespace().count();

            // If there is a trailing delimiter from the previous field, push it
            if let Some(del) = trailing_delimiter {
                formatted_str = del.to_string();
            };

            // If the current field isn't empty, set trailing delimiter for
            // upcoming loop repitition. If it's empty, start next run of loop
            // directly
            if !field.is_empty() {
                trailing_delimiter = cur_trailing_delimiter;
            } else {
                continue 'field_loop;
            }

            // loop over single parts of current field and add correct delimiter
            // process the single slices and add correct delimiter
            'word_loop: for (idx, field_slice) in split_field.enumerate() {
                // if the current slice is a common word from the ignore list,
                // skip it.
                if ignored_words.contains(&field_slice.to_lowercase()) {
                    continue;
                }

                // Create word slice char by char. We need to loop over chars
                // instead of a simple bytes index to also catch chars which
                // consist of more than one byte (äöüøæ etc...)
                let mut word_slice = String::new();
                let word_chars = field_slice.chars();
                let mut counter = 0;
                'char_loop: for mut c in word_chars {
                    // If camelcase is set, force first char of word to uppercase
                    if counter == 0 && case == Some(&CitekeyCase::Camel) {
                        c = c.to_ascii_uppercase()
                    }
                    // if non-ascii chars should be mapped, check if needed and do it
                    if let Some(chars) = deunicode::deunicode_char(c)
                        && ascii_only
                    {
                        'deunicode_char_loop: for ch in chars.chars() {
                            // if the deunicoded charset of the word slice's current
                            // char contains a special char, skip it
                            if ignored_chars.contains(&ch) {
                                continue 'deunicode_char_loop;
                            }
                            word_slice.push(ch);
                            counter += 1;
                            if let Some(len) = max_chars
                                && counter >= len
                            {
                                break 'char_loop;
                            }
                        }
                    } else {
                        // if a word slice contains a special char, skip it
                        if ignored_chars.contains(&c) {
                            continue 'char_loop;
                        }
                        word_slice.push(c);
                        counter += 1;
                        if let Some(len) = max_chars
                            && counter >= len
                        {
                            break 'char_loop;
                        }
                    }
                }
                // Don't count empty slices and don't add delimiter to those
                if !word_slice.is_empty() {
                    formatted_str = formatted_str + &word_slice;
                    words_passed += 1;
                    if max_words.is_some_and(|max| max == words_passed) || idx + 1 == field_count {
                        break 'word_loop;
                    } else {
                        formatted_str = formatted_str + inner_delimiter.unwrap_or("");
                    }
                } else {
                    continue 'word_loop;
                }
            }
            formatted_str
        };
        new_citekey = new_citekey + &formatted_field_str;
    }
    match case {
        Some(CitekeyCase::Lower) => new_citekey.to_lowercase(),
        Some(CitekeyCase::Upper) => new_citekey.to_uppercase(),
        // otherwise skip, since camelcase is processed in char loop
        _ => new_citekey,
    }
}

/// Preformat some fields which are very common to be used in citekeys
pub(super) fn preformat_field(field: &str, entry: &Entry) -> String {
    match field {
        // Sanitize all macro code from string
        "title" => {
            sanitize_single_string_fully(&entry.get_as::<String>(field).unwrap_or("".into()))
        }
        // Get author names. Fall back to editors before setting empty string
        "author" => {
            if let Ok(authors) = entry.author() {
                let mut last_names = String::new();
                for a in authors.iter() {
                    last_names = last_names + &a.name + " ";
                }
                last_names
            } else if let Ok(editors) = entry.editors() {
                let mut last_names = String::new();
                for editortypes in editors.iter() {
                    for e in editortypes.0.iter() {
                        last_names = last_names + &e.name + " ";
                    }
                }
                last_names
            } else {
                "".to_string()
            }
        }
        // Get year of date field, fallback to year field
        "year" => {
            if let Ok(date) = entry.date() {
                date.to_chunks().format_verbatim()[..4].to_string()
            } else {
                entry.get_as::<String>(field).unwrap_or("".into())
            }
        }
        // Sanitize all macro code from string
        "subtitle" => {
            sanitize_single_string_fully(&entry.get_as::<String>(field).unwrap_or("".into()))
        }
        "editor" => {
            if let Ok(editors) = entry.editors() {
                let mut last_names = String::new();
                for editortypes in editors.iter() {
                    for e in editortypes.0.iter() {
                        last_names = last_names + &e.name + " ";
                    }
                }
                last_names
            } else {
                "".to_string()
            }
        }
        "pubtype" | "entrytype" => entry.entry_type.to_string(),
        _ => entry.get_as::<String>(field).unwrap_or("".into()),
    }
}

/// Split a formatting pattern of kind
/// `<field>;<word count>;<char count>;<inside delimiter>;<trailing delimiter>`,
/// e.g.: `title;3;3;_;:` will give `("title", 3, 3, "_", ":")`
pub(super) fn split_formatting_pat(
    pattern: &str,
) -> (
    &str,
    Option<usize>,
    Option<usize>,
    Option<&str>,
    Option<&str>,
) {
    let mut splits = pattern.split(';');
    (
        splits
            .next()
            .expect("Need field value for formatting citekey"),
        if let Some(next) = splits.next()
            && next.len() > 0
        {
            next.parse::<usize>().ok()
        } else {
            None
        },
        if let Some(next) = splits.next()
            && next.len() > 0
        {
            next.parse::<usize>().ok()
        } else {
            None
        },
        splits.next(),
        splits.next(),
    )
}

#[cfg(test)]
mod test {
    use crate::bibiman::citekeys::citekey_utils::split_formatting_pat;

    #[test]
    fn split_citekey_pattern() {
        let pattern = "title;3;5;_;_";

        assert_eq!(
            split_formatting_pat(pattern),
            ("title", Some(3), Some(5), Some("_"), Some("_"))
        );

        let pattern = "year";

        assert_eq!(
            split_formatting_pat(pattern),
            ("year", None, None, None, None)
        );

        let pattern = "author;1;;;_";
        assert_eq!(
            split_formatting_pat(pattern),
            ("author", Some(1), None, Some(""), Some("_"))
        );
    }
}
