#!/usr/bin/env bash

load_colors() {
    col1='\033[92m'
    col2='\033[94m'
    boldtxt='\033[1m'
    warn='\033[91m'
    reset='\033[0m'
}

setup_vars() {
    # Set up the variables containing the directories for installation
    # depending if the script was run as root or local user.
    # Dirs for systemwide/global installation
    BINDIR="${PREFIX}/bin"
    MANDIR="${PREFIX}/share/man"
    APPLICATIONSDIR="${PREFIX}/share/applications"
    if zsh --version >/dev/null 2>&1; then
        ZSHCOMPDIR="${PREFIX}/share/zsh/site-functions"
    fi
    if pkg-config --variable=completionsdir bash-completion >/dev/null 2>&1; then
        BASHCOMPDIR="$(pkg-config --variable=completionsdir bash-completion)"
    fi
}

install_bibiman() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
        "--prefix")
            shift
            if [[ $# -gt 0 ]]; then
                PREFIX="$1"
                shift
            else
                printf "${warn}${boldtxt}Error:${reset} ${boldtxt}--prefix${reset} option needs argument\n"
                exit 1
            fi
            ;;
        *)
            printf "Unknown argument ${warn}%s$reset, aborting. Showing the help:\n\n" "$1"
            usage
            exit 1
            ;;
        esac
    done

    setup_vars

    install -D -m 755 ./target/release/bibiman "$BINDIR"/bibiman
    printf "Installed ${col1}${boldtxt}bibiman${reset} to ${col2}%s${reset}\n" "$BINDIR"
    install -D -m 644 ./man/bibiman.1 "$MANDIR"/man1/bibiman.1
    printf "Installed ${boldtxt}bibiman.1${reset} to ${col2}%s${reset}\n" "$MANDIR/man1"
    install -D -m 644 ./man/bibiman.toml.5 "$MANDIR"/man5/bibiman.toml.5
    printf "Installed ${boldtxt}bibiman.toml.5${reset} to ${col2}%s${reset}\n\n" "$MANDIR/man5"
    printf "You might need to run \`${boldtxt}sudo mandb${reset}\` to update the manpage database\n"
}

print_vars() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
        "--prefix")
            shift
            if [[ $# -gt 0 ]]; then
                PREFIX="$1"
                shift
            else
                printf "${warn}${boldtxt}Error:${reset} ${boldtxt}--prefix${reset} option needs argument\n"
                exit 1
            fi
            ;;
        *)
            printf "Unknown argument ${warn}%s$reset, aborting. Showing the help:\n\n" "$1"
            usage
            exit 1
            ;;
        esac
    done

    setup_vars

    echo "PREFIX: $PREFIX"
    echo "BINDIR: $BINDIR"
    echo "MANDIR: $MANDIR"
    echo "APPLICATIONSDIR: $APPLICATIONSDIR"
    echo "ZSHCOMPDIR: $ZSHCOMPDIR"
    echo "BASHCOMPDIR: $BASHCOMPDIR"
}

usage() {
    printf "Install script for the bibiman Git repository\n\n"
    printf "USAGE: ./install.sh [install|uninstall|print-vars] [OPTIONS]\n\n"
    printf "OPTIONS\n"
    printf "\t${boldtxt}--prefix <PATH>${reset}\n"
    printf "\tSet the prefix for installing the binary and additional ressources.\n"
    printf "\tDefault is ${boldtxt}/usr/local${reset}.\n"
    printf "\tExample: use ${boldtxt}--prefix %s${reset} to install locally\n\n" "$HOME/.local"
}

uninstall_bibiman() {
    while [[ $# -gt 0 ]]; do
        case "$1" in
        "--prefix")
            shift
            if [[ $# -gt 0 ]]; then
                PREFIX="$1"
                shift
            else
                printf "${warn}${boldtxt}Error:${reset} ${boldtxt}--prefix${reset} option needs argument\n"
                exit 1
            fi
            ;;
        *)
            printf "Unknown argument ${warn}%s$reset, aborting. Showing the help:\n\n" "$1"
            usage
            exit 1
            ;;
        esac
    done

    setup_vars

    if [[ -f "$BINDIR/bibiman" ]]; then
        rm -f "$BINDIR/bibiman"
        printf "${warn}Removed${reset} ${col2}${boldtxt}%s${reset}\n" "$BINDIR/bibiman"
    else
        printf "$BINDIR/bibiman ${warn}not found${reset}\n"
    fi

    if [[ -f "$MANDIR/man1/bibiman.1" ]]; then
        rm -f "$MANDIR/man1/bibiman.1"
        printf "${warn}Removed${reset} ${col2}${boldtxt}%s${reset}\n" "$MANDIR/man1/bibiman.1"
    else
        printf "$MANDIR/man1/bibiman.1 ${warn}not found${reset}\n"
    fi

    if [[ -f "$MANDIR/man5/bibiman.toml.5" ]]; then
        rm -f "$MANDIR/man5/bibiman.toml.5"
        printf "${warn}Removed${reset} ${col2}${boldtxt}%s${reset}\n" "$MANDIR/man5/bibiman.toml.5"
    else
        printf "$MANDIR/man5/bibiman.toml.5 ${warn}not found${reset}\n"
    fi

}

main() {
    set -eu

    load_colors

    if [[ $# -eq 0 ]]; then
        printf "You did not pass any argument to the script. See the help:\n\n"
        usage
        exit 1
    fi

    PREFIX="/usr/local"

    if [[ $1 == install ]]; then
        shift
        install_bibiman "$@"
    elif [[ $1 == uninstall ]]; then
        shift
        uninstall_bibiman "$@"
    elif [[ $1 == "print-vars" ]]; then
        shift
        print_vars "$@"
    elif [[ $1 == --help || $1 == -h ]]; then
        usage
        exit 0
    else
        printf "Unknown argument ${warn}%s$reset, aborting. Showing the help:\n\n" "$1"
        usage
        exit 1
    fi
}

main "$@"
